% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function-isotope.R
\name{isotopologues}
\alias{isotopologues}
\title{Identfying isotopologue peaks in MS data}
\usage{
isotopologues(
  x,
  substDefinition = isotopicSubstitutionMatrix(),
  tolerance = 0,
  ppm = 20,
  seedMz = numeric(),
  charge = 1,
  .check = TRUE
)
}
\arguments{
\item{x}{\code{matrix} or \code{data.frame} with spectrum data. The first column is
expected to contain \emph{m/z} and the second column intensity values. The
\emph{m/z} values in that matrix are expected to be increasingly ordered
and no \code{NA} values should be present.}

\item{substDefinition}{\code{matrix} or \code{data.frame} with definition of isotopic
substitutions (columns \code{"name"} and \code{"md"} are among the mandatory
columns). The rows in this matrix have to be ordered by column \code{md}
in increasing order. See \code{\link[=isotopicSubstitutionMatrix]{isotopicSubstitutionMatrix()}} for more
information on the format and content.}

\item{tolerance}{\code{numeric(1)} representing the absolute tolerance for the
relaxed matching of m/z values of peaks. See \code{\link[MsCoreUtils:matching]{MsCoreUtils::closest()}} for
details.}

\item{ppm}{\code{numeric(1)} representing a relative, value-specific
parts-per-million (PPM) tolerance for the relaxed matching of m/z values
of peaks. See \code{\link[MsCoreUtils:matching]{MsCoreUtils::closest()}} for details.}

\item{seedMz}{\code{numeric} vector of \strong{ordered} m/z values. If provided,
the function checks if there are peaks in \code{x} which m/z match them.
If any, it looks for groups where the first peak is one of the matched
ones.}

\item{charge}{\code{numeric(1)} representing the expected charge of the ionized
compounds.}

\item{.check}{\code{logical(1)} to disable input argument check. Should only be
set to \code{FALSE} if provided \emph{m/z} values are guaranteed to be increasingly
ordered and don't contain \code{NA} values.}
}
\value{
\code{list} of \code{integer} vectors. Each \code{integer} vector contains the
indixes of the rows in \code{x} with potential isotopologues of the same
compound.
}
\description{
Given a spectrum (i.e. a peak matrix with m/z and intensity values)
the function identifies groups of potential isotopologue peaks based on
pre-defined mass differences and intensity (probability) ratios that need
to be passed to the function with the \code{substDefinition} parameter. Each
isotopic substitution in a compound determines a certain isotopologue and it
is associated with a certain mass difference of that with respect to the
monoisotopic isotopologue. Also each substitution in a compound is linked
to a certain ratio between the intensities of the peaks of the corresponding
isotopologue and the monoisotopic one. This ratio isn't the same for
isotopologues corresponding to the same isotopic substitution but to
different compounds. Through the \code{substDefinition} parameter we provide
upper and lower values to compute bounds for each isotopic substitution
dependent on the peak's mass.
}
\details{
The function iterates over the peaks (rows) in \code{x}. For each peak (which is
assumed to be the monoisotopic peak) it searches other peaks in \code{x} with a
difference in mass matching (given \code{ppm} and \code{tolerance}) any of the
pre-defined mass differences in \code{substDefinitions} (column \code{"md"}). The mass
is obtained by multiplying the m/z of the peaks for the \code{charge} expected
for the ionized compounds.

For matching peaks, the function next evaluates whether their intensity is
within the expected (pre-defined) intensity range. Using \code{"LBint"},
\code{"LBslope"}, \code{"UBint"}, \code{"UBslope"} of the previously matched isotopic
substitution in \code{substDefinition}, the function estimates a (mass dependent)
lower and upper intensity ratio limit based on the peak's mass.

When some peaks are grouped together their indexes are excluded from the set
of indexes that are searched for further groups (i.e. peaks already assigned
to an isotopologue group are not considered/tested again thus each peak can
only be part of one isotopologue group).
}
\examples{

## Read theoretical isotope pattern (high resolution) from example file
x <- read.table(system.file("exampleSpectra",
    "serine-alpha-lactose-caffeine.txt", package = "MetaboCoreUtils"),
    header = TRUE)
x <- x[order(x$mz), ]
plot(x$mz, x$intensity, type = "h")

isos <- isotopologues(x, ppm = 5)
isos

## highlight them in the plot
for (i in seq_along(isos)) {
    z <- isos[[i]]
    points(x$mz[z], x$intensity[z], col = i + 1)
}
}
\author{
Andrea Vicini
}
