% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normalization.R
\name{fit_lm}
\alias{fit_lm}
\alias{adjust_lm}
\title{Linear model-based normalization of abundance matrices}
\usage{
fit_lm(
  formula,
  data,
  y,
  method = c("lm", "lmrob"),
  control = NULL,
  minVals = ceiling(nrow(data) * 0.75),
  model = TRUE,
  ...,
  BPPARAM = SerialParam()
)

adjust_lm(y = matrix(), data = data.frame(), lm = list(), ...)
}
\arguments{
\item{formula}{\code{formula} defining the model that should be fitted to the
data. See also \code{\link[=lm]{lm()}} for more information. Formulas should begin
with \verb{y ~ } as values in rows of \code{y} will be defined as \emph{y}. See
description of the \code{fit_lm} function for more information.}

\item{data}{\code{data.frame} containing the covariates for the linear model
defined by \code{formula} (for \code{fit_lm}) or used in \code{lm} (for \code{adjust_lm}).
The number of rows has to match the number of columns of \code{y}.}

\item{y}{for \code{fit_lm}: \code{matrix} of abundances on which the linear model
pefined with \code{formula} should be fitted. For \code{adjust_lm}: \code{matrix}
of abundances that should be adjusted using the models provided with
parameter \code{lm}.}

\item{method}{\code{character(1)} defining the linear regression function that
should be used for model fitting. Can be either \code{method = "lm"} (the
default) for standard least squares model fitting or \code{method = "lmrob"}
for a robust alternative defined in the \emph{robustbase} package.}

\item{control}{a \code{list} speficying control parameters for \code{lmrob}. Only
used if \code{method = "lmrob"}. See help of \code{lmrob.control} in the
\code{robustbase} package for details. By default \code{control = NULL} the
\emph{KS2014} settings are used and scale-finding iterations are increased
to 10000.}

\item{minVals}{\code{numeric(1)} defining the minimum number of non-missing
values (per feature/row) required to perform the model fitting. For
rows in \code{y} for which fewer non-\code{NA} values are available no model
will be fitted and a \code{NA} will be reported instead.}

\item{model}{\code{logical(1)} whether the model frame are included in the
returned linear models. Passed to the \code{lm} or \code{lmrob} functions.}

\item{...}{for \code{fit_lm}: additional parameters to be passed to the
downstream calls to \code{lm} or \code{lmrob}. For \code{adjust_lm}: ignored.}

\item{BPPARAM}{parallel processing setup. See \code{\link[BiocParallel:register]{BiocParallel::bpparam()}} for more
information. Parallel processing can improve performance especially
for \code{method = "lmrob"}.}

\item{lm}{\code{list} of linear models (as returned by \code{lm} or \code{lmrob}) such
as returned by the \code{fit_lm} function. The length of the list is
expected to match the number of rows of \code{y}, i.e., each element
should be a linear model to adjust the specific row, or \code{NA} to skip
adjustment for that particular row in \code{y}.}
}
\value{
\if{html}{\out{<div class="sourceCode">}}\preformatted{For `fit_lm`: a `list` with linear models (either of type *lm* or
*lmrob*) or length equal to the number of rows of `y`. `NA` is
reported for rows with too few non-missing data points (depending
on parameter `minValues`).
For `adjust_lm`: a numeric matrix (same dimensions as input matrix
`y`) with the values adjusted with the provided linear models.
}\if{html}{\out{</div>}}
}
\description{
The \code{fit_lm} and \code{adjust_lm} functions facilitate linear model-based
normalization of abundance matrices. The expected noise in a numeric
data matrix can be modeled with a linear regression model using the
\code{fit_lm} function and the data can subsequently be adjusted using the
\code{adjust_lm} function (i.e., the modeled noise will be removed from the
abundance values). A typical use case would be to remove injection index
dependent signal drifts in a LC-MS derived metabolomics data:
a linear model of the form \code{y ~ injection_index} could be used to model
the measured abundances of each feature (each row in a data matrix) as a
function of the injection index in which a specific sample was measured
during the LC-MS measurement run. The fitted linear regression models
can subsequently be used to adjust the abundance matrix by removing
these dependencies from the data. This allows to perform signal
adjustments as described in (Wehrens et al. 2016).

The two functions are described in more details below:

\code{fit_lm} allows to fit a linear regression model (defined with parameter
\code{formula}) to each row of the numeric data matrix submitted with parameter
\code{y}. Additional covariates of the linear model defined in \code{formula} are
expected to be provided as columns in a \code{data.frame} supplied \emph{via}
the \code{data} parameter.

The linear model is expected to be defined by a formula starting with
\verb{y ~ }. To model for example an injection index dependency of values in
\code{y} a formula \code{y ~ injection_index} could be used, with values for the
injection index being provided as a column \code{"injection_index"} in the
\code{data} data frame. \code{fit_lm} would thus fit this model to each row of
\code{y}.

Linear models can be fitted either with the standard least squares of
\code{\link[=lm]{lm()}} by setting \code{method = "lm"} (the default), or with the more robust
methods from the \emph{robustbase} package with \code{method = "lmrob"}.

\code{adjust_lm} can be used to adjust abundances in a data matrix \code{y} based
on linear regression models provided with parameter \code{lm}. Parameter \code{lm}
is expected to be a \code{list} of length equal to the number of rows of \code{y},
each element being a linear model (i.e., a results from \code{lm} or \code{lmrob}).
Covariates for the model need to be provided as columns in a \code{data.frame}
provided with parameter \code{data}. The number of rows of that \code{data.frame}
need to match the number of columns of \code{y}. The function returns the
input matrix \code{y} with values in rows adjusted with the linear models
provided by \code{lm}. No adjustment is performed for rows for which the
respective element in \code{lm} is \code{NA}. See examples below for details or the
vignette for more examples, descriptions and information.
}
\examples{

## See also the vignette for more details and examples.

## Load a test matrix with abundances of features from a LC-MS experiment.
vals <- read.table(system.file("txt", "feature_values.txt",
                                package = "MetaboCoreUtils"), sep = "\t")
vals <- as.matrix(vals)

## Define a data.frame with the covariates to be used to model the noise
sdata <- data.frame(injection_index = seq_len(ncol(vals)))

## Fit a linear model describing the feature abundances as a
## function of the index in which samples were injected during the LC-MS
## run. We're fitting the model to log2 transformed data.
## Note that such a model should **only** be fitted if the samples
## were randomized, i.e. the injection index is independent of any
## experimental covariate. Alternatively, the injection order dependent
## signal drift could be estimated using QC samples (if they were
## repeatedly injected) - see vignette for more details.
ii_lm <- fit_lm(y ~ injection_index, data = sdata, y = log2(vals))

## The result is a list of linear models
ii_lm[[1]]

## Plotting the data for one feature:
plot(x = sdata$injection_index, y = log2(vals[2, ]),
    ylab = expression(log[2]~abundance), xlab = "injection index")
grid()
## plot also the fitted model
abline(ii_lm[[2]], lty = 2)

## For this feature (row) a decreasing signal intensity with injection
## index was observed (and modeled).

## For another feature an increasing intensity can be observed.
plot(x = sdata$injection_index, y = log2(vals[3, ]),
    ylab = expression(log[2]~abundance), xlab = "injection index")
grid()
## plot also the fitted model
abline(ii_lm[[3]], lty = 2)

## This trend can be removed from the data using the `adjust_lm` function
## by providing the linear models descring the drift. Note that, because
## we're adjusting log2 transformed data, the resulting abundances are
## also in log2 scale.
vals_adj <- adjust_lm(log2(vals), data = sdata, lm = ii_lm)

## Plotting the data before (open circles) and after adjustment (filled
## points)
plot(x = sdata$injection_index, y = log2(vals[2, ]),
    ylab = expression(log[2]~abundance), xlab = "injection index")
points(x = sdata$injection_index, y = vals_adj[2, ], pch = 16)
grid()
## Adding the line fitted through the raw data
abline(ii_lm[[2]], lty = 2)
## Adding a line fitted through the adjusted data
abline(lm(vals_adj[2, ] ~ sdata$injection_index), lty = 1)
## After adjustment there is no more dependency on injection index.
}
\references{
Wehrens R, Hageman JA, van Eeuwijk F, Kooke R, Flood PJ, Wijnker E,
Keurentjes JJ, Lommen A, van Eekelen HD, Hall RD Mumm R and de Vos RC.
Improved batch correction in untargeted MS-based metabolomics.
\emph{Metabolomics} 2016; 12:88.
}
\author{
Johannes Rainer
}
