% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MatchedSpectra.R
\name{MatchedSpectra}
\alias{MatchedSpectra}
\alias{MatchedSpectra-class}
\alias{spectraVariables,MatchedSpectra-method}
\alias{queryVariables,MatchedSpectra-method}
\alias{targetVariables,MatchedSpectra-method}
\alias{colnames,MatchedSpectra-method}
\alias{$,MatchedSpectra-method}
\alias{spectraData,MatchedSpectra-method}
\alias{matchedData,MatchedSpectra-method}
\alias{addProcessing,MatchedSpectra-method}
\alias{plotSpectraMirror,MatchedSpectra-method}
\alias{setBackend,MatchedSpectra,MsBackend-method}
\title{Representation of Spectra matches}
\usage{
MatchedSpectra(
  query = Spectra(),
  target = Spectra(),
  matches = data.frame(query_idx = integer(), target_idx = integer(), score = numeric())
)

\S4method{spectraVariables}{MatchedSpectra}(object)

\S4method{queryVariables}{MatchedSpectra}(object)

\S4method{targetVariables}{MatchedSpectra}(object)

\S4method{colnames}{MatchedSpectra}(x)

\S4method{$}{MatchedSpectra}(x, name)

\S4method{spectraData}{MatchedSpectra}(object, columns = spectraVariables(object))

\S4method{matchedData}{MatchedSpectra}(object, columns = spectraVariables(object), ...)

\S4method{addProcessing}{MatchedSpectra}(object, FUN, ..., spectraVariables = character())

\S4method{plotSpectraMirror}{MatchedSpectra}(
  x,
  xlab = "m/z",
  ylab = "intensity",
  main = "",
  scalePeaks = FALSE,
  ...
)

\S4method{setBackend}{MatchedSpectra,MsBackend}(object, backend, ...)
}
\arguments{
\item{query}{\code{Spectra} with the query spectra.}

\item{target}{\code{Spectra} with the spectra against which \code{query} has been
matched.}

\item{matches}{\code{data.frame} with columns \code{"query_idx"} (\code{integer}),
\code{"target_idx"} (\code{integer}) and \code{"score"} (\code{numeric}) representing the
\emph{n:m} mapping of elements between the \code{query} and the \code{target} \code{Spectra}.}

\item{object}{\code{MatchedSpectra} object.}

\item{x}{\code{MatchedSpectra} object.}

\item{name}{for \code{$}: the name of the spectra variable to extract.}

\item{columns}{for \code{spectraData}: \code{character} vector with spectra variable
names that should be extracted.}

\item{...}{for \code{addProcessing}: additional parameters for the function \code{FUN}.
For \code{plotSpectraMirror}: additional parameters passed to the plotting
functions.}

\item{FUN}{for \code{addProcessing}: function to be applied to the peak matrix
of each spectrum in \code{object}. See \code{\link[Spectra:Spectra]{Spectra::Spectra()}} for more details.}

\item{spectraVariables}{for \code{addProcessing}: \code{character} with additional
spectra variables that should be passed along to the function defined
with \code{FUN}. See \code{\link[Spectra:Spectra]{Spectra::Spectra()}} for details.}

\item{xlab}{for \code{plotSpectraMirror}: the label for the x-axis.}

\item{ylab}{for \code{plotSpectraMirror}: the label for the y-axis.}

\item{main}{for \code{plotSpectraMirror}: an optional title for each plot.}

\item{scalePeaks}{for \code{plotSpectraMirror}: \code{logical(1)} if peak intensities
(per spectrum) should be scaled to a total sum of one (per spectrum) prior
to plotting.}

\item{backend}{for \code{setBackend}: instance of an object extending
\link[Spectra:MsBackend]{Spectra::MsBackend}. See help for \code{\link[Spectra:Spectra]{Spectra::setBackend()}}
for more details.}
}
\value{
See individual method desciption above for details.
}
\description{
Matches between query and target spectra can be represented by the
\code{MatchedSpectra} object. Functions like the \code{\link[=matchSpectra]{matchSpectra()}} function will
return this type of object. By default, all data accessors work as
\emph{left joins} between the \emph{query} and the \emph{target} spectra, i.e. values are
returned for each \emph{query} spectrum with eventual duplicated entries (values)
if the query spectrum matches more than one target spectrum.
}
\section{Creation, subset and filtering}{


\code{MatchedSpectra} objects are the result object from the \code{\link[=matchSpectra]{matchSpectra()}}.
While generally not needed, \code{MatchedSpectra} objects can also be created
with the \code{MatchedSpectra} function providing the \code{query} and \code{target}
\code{Spectra} objects as well as a \code{data.frame} with the \emph{matches} between
query and target elements. This data frame is expected to have columns
\code{"query_idx"}, \code{"target_idx"} with the \code{integer} indices of query and
target objects that are \emph{matched} and a column \code{"score"} with a \code{numeric}
score for the match.

\code{MatchedSpectra} objects can be subset using:
\itemize{
\item \code{[} subset the \code{MatchedSpectra} selecting \code{query} spectra to keep with
parameter \code{i}. The \code{target} spectra will by default be returned as-is.
\item \code{pruneTarget} \emph{cleans} the \code{MatchedSpectra} object by removing non-matched
target spectra.
}

In addition, \code{MatchedSpectra} can be filtered with any of the filtering
approaches defined for \code{\link[=Matched]{Matched()}} objects: \code{\link[=SelectMatchesParam]{SelectMatchesParam()}},
\code{\link[=TopRankedMatchesParam]{TopRankedMatchesParam()}} or \code{\link[=ScoreThresholdParam]{ScoreThresholdParam()}}.
}

\section{Extracting data}{

\itemize{
\item \code{$} extracts a single spectra variable from the \code{MatchedSpectra} \code{x}. Use
\code{spectraVariables} to get all available spectra variables. Prefix
\code{"target_"} is used for spectra variables from the \emph{target} \code{Spectra}. The
matching scores are available as \emph{spectra variable} \code{"score"}.
Similar to a left join between the query and target spectra, this function
returns a value for each query spectrum with eventual duplicated values for
query spectra matching more than one target spectrum. If spectra variables
from the target spectra are extracted, an \code{NA} is reported for \emph{query}
spectra that don't match any target spectra. See examples below for more
details.
\item \code{length} returns the number of \strong{query} spectra.
\item \code{matchedData} same as \code{spectraData} below.
\item \code{query} returns the \emph{query} \code{Spectra}.
\item \code{queryVariables} returns the \code{spectraVariables} of \emph{query}.
\item \code{spectraData} returns spectra variables from the query and/or target
\code{Spectra} as a \code{DataFrame}. Parameter \code{columns} allows to define which
variables should be returned (defaults to
\code{columns = spectraVariables(object)}), spectra variable names of the target
spectra need to be prefixed with \code{target_} (e.g. \code{target_msLevel} to get
the MS level from target spectra). The score from the matching function is
returned as spectra variable \code{"score"}. Similar to \code{$}, this function
performs a \emph{left join} of spectra variables from the \emph{query} and \emph{target}
spectra returning all values for all query spectra (eventually returning
duplicated elements for query spectra matching multiple target spectra)
and the values for the target spectra matched to the respective query
spectra. See help on \code{$} above or examples below for details.
\item \code{spectraVariables} returns all available spectra variables in the \emph{query}
and \emph{target} spectra. The prefix \code{"target_"} is used to label spectra
variables of target spectra (e.g. the name of the spectra variable for the
MS level of target spectra is called \code{"target_msLevel"}).
\item \code{target} returns the \emph{target} \code{Spectra}.
\item \code{targetVariables} returns the \code{spectraVariables} of \emph{target} (prefixed
with \code{"target_"}).
\item \code{whichTarget} returns an \code{integer} with the indices of the spectra in
\emph{target} that match at least on spectrum in \emph{query}.
\item \code{whichQuery} returns an \code{integer} with the indices of the spectra in
\emph{query} that match at least on spectrum in \emph{target}.
}
}

\section{Data manipulation and plotting}{

\itemize{
\item \code{addProcessing}: add a processing step to both the \emph{query} and \emph{target}
\code{Spectra} in \code{object}. Additional parameters for \code{FUN} can be passed \emph{via}
\code{...}. See \code{addProcessing} documentation in \code{\link[Spectra:Spectra]{Spectra::Spectra()}} for more
information.
\item \code{plotSpectraMirror}: creates a mirror plot between the query and each
matching target spectrum. Can only be applied to a \code{MatchedSpectra} with a
single query spectrum. Setting parameter \code{scalePeaks = TRUE} will scale
the peak intensities per spectrum to a total sum of one for a better
graphical visualization. Additional plotting parameters can be passed
through \code{...}. The parameters \code{ppm} and \code{tolerance} can be used to
define the m/z tolerance for matching peaks between the query and target
spectra. If not provided by the user, the values from the \code{param}
object used to create the \code{MatchedSpectra} object are used; if these are
missing, the default values (\code{ppm =20} and \code{tolerance = 0}) are used.
\item \code{setBackend}: allows to change the \emph{backend} of both the query and target
\code{\link[Spectra:Spectra]{Spectra::Spectra()}} object. The function will return a \code{MatchedSpectra}
object with the query and target \code{Spectra} changed to the specified
\code{backend}, which can be any backend extending \link[Spectra:MsBackend]{Spectra::MsBackend}.
}
}

\examples{

## Creating a dummy MatchedSpectra object.
library(Spectra)
df1 <- DataFrame(
    msLevel = 2L, rtime = 1:10,
    spectrum_id = c("a", "b", "c", "d", "e", "f", "g", "h", "i", "j"))
df2 <- DataFrame(
    msLevel = 2L, rtime = rep(1:10, 20),
    spectrum_id = rep(c("A", "B", "C", "D", "E"), 20))
sp1 <- Spectra(df1)
sp2 <- Spectra(df2)
## Define matches between query spectrum 1 with target spectra 2 and 5,
## query spectrum 2 with target spectrum 2 and query spectrum 4 with target
## spectra 8, 12 and 15.
ms <- MatchedSpectra(
    sp1, sp2, matches = data.frame(query_idx = c(1L, 1L, 2L, 4L, 4L, 4L),
                                   target_idx = c(2L, 5L, 2L, 8L, 12L, 15L),
                                   score = 1:6))

## Which of the query spectra match at least one target spectrum?
whichQuery(ms)

## Extracting spectra variables: accessor methods for spectra variables act
## as "left joins", i.e. they return a value for each query spectrum, with
## eventually duplicated elements if one query spectrum matches more than
## one target spectrum.

## Which target spectrum matches at least one query spectrum?
whichTarget(ms)

## Extracting the retention times of the query spectra.
ms$rtime

## We have duplicated retention times for query spectrum 1 (matches 2 target
## spectra) and 4 (matches 3 target spectra). The retention time is returned
## for each query spectrum.

## Extracting retention times of the target spectra. Note that only retention
## times for target spectra matching at least one query spectrum are returned
## and an NA is reported for query spectra without matching target spectrum.
ms$target_rtime

## The first query spectrum matches target spectra 2 and 5, thus their
## retention times are returned as well as the retention time of the second
## target spectrum that matches also query spectrum 2. The 3rd query spectrum
## does match any target spectrum, thus `NA` is returned. Query spectrum 4
## matches target spectra 8, 12, and 15, thus the next reported retention
## times are those from these 3 target spectra. None of the remaining 6 query
## spectra matches any target spectra and thus `NA` is reported for each of
## them.

## With `queryIndex` and `targetIndex` it is possible to extract the indices
## of the matched query-index pairs
queryIndex(ms)
targetIndex(ms)

## The first match is between query index 1 and target index 2, the second
## match between query index 1 and target index 5 and so on.
## We could use these indices to extract a `Spectra` object containing only
## matched target spectra and assign a spectra variable with the indices of
## the query spectra
matched_target <- target(ms)[targetIndex(ms)]
matched_target$query_index <- queryIndex(ms)

## This `Spectra` object thus contains information from the matching, but
## is a *conventional* `Spectra` object that could be used for further
## analyses.

## `spectraData` can be used to extract all (or selected) spectra variables
## from the object. Same as with `$`, a left join between the specta
## variables from the query spectra and the target spectra is performed. The
## prefix `"target_"` is used to label the spectra variables from the target
## spectra. Below we extract selected spectra variables from the object.
res <- spectraData(ms, columns = c("rtime", "spectrum_id",
    "target_rtime", "target_spectrum_id"))
res
res$spectrum_id
res$target_spectrum_id

## Again, all values for query spectra are returned and for query spectra not
## matching any target spectrum NA is reported as value for the respecive
## variable.

## The example matched spectra object contains all query and all target
## spectra. Below we subset the object keeping only query spectra that are
## matched to at least one target spectrum.
ms_sub <- ms[whichQuery(ms)]

## ms_sub contains now only 3 query spectra:
length(query(ms_sub))

## while the original object contains all 10 query spectra:
length(query(ms))

## Both object contain however still the full target `Spectra`:
length(target(ms))
length(target(ms_sub))

## With the `pruneTarget` we can however reduce also the target spectra to
## only those that match at least one query spectrum
ms_sub <- pruneTarget(ms_sub)
length(target(ms_sub))
}
\seealso{
\code{\link[=Matched]{Matched()}} for additional functions available for \code{MatchedSpectra}.
}
\author{
Johannes Rainer
}
