% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prior_modelling.R
\name{prior_combined}
\alias{prior_combined}
\title{Prior Function for Model Components}
\usage{
prior_combined(params, component, y, x_vars, z, use_data_priors = TRUE, user_fixed_priors)
}
\arguments{
\item{params}{A numeric vector of parameters for the model. Typically, \code{params} will include a set of parameters
associated with the regression coefficients for the model. For example, \code{params} might be
\eqn{(a, b, c, d, e)} representing coefficients for the covariates in \code{x_vars}.}

\item{component}{An integer specifying which component of the model to consider. The model is a mixture with multiple
components, this argument indicates which component’s parameters and priors are being evaluated.}

\item{y}{A numeric vector of observed responses (e.g., interaction counts). This is used when estimating data-driven
priors. The vector should be of length \code{N}, where \code{N} is the size of the system.}

\item{x_vars}{A list of covariate information. Each element of \code{x_vars} is typically another list or vector
representing a single covariate. These covariates are used to compute data-driven priors by extracting summary statistics
from the data corresponding to the specified component.}

\item{z}{A matrix or similar structure used to assign observations to components. If \code{z == component}, that observation
belongs to the given component. The function uses this to subset \code{y} and \code{x_vars} for calculating data-driven priors.}

\item{use_data_priors}{A logical value indicating whether to use data-driven priors (\code{TRUE}) or user-specified fixed priors (\code{FALSE}).
When \code{TRUE}, the function computes means and standard deviations from the subset of the data assigned to the specified component.}

\item{user_fixed_priors}{A list of user-defined priors for each component. This argument is only used when
\code{use_data_priors = FALSE}. The list should contain entries named \code{component1}, \code{component2}, and \code{component3},
each defining \code{meany}, \code{meanx1}, \code{meanx2}, \code{meanx3}, \code{meanx4},
and corresponding \code{sdy}, \code{sdx1}, \code{sdx2}, \code{sdx3}, \code{sdx4}.}
}
\value{
A single numeric value representing the sum of the log-priors for all parameters in \code{params}.
This value can be integrated into a Bayesian inference framework to update parameter estimates based on the chosen priors.
}
\description{
This function computes the prior contribution to the log-posterior for the parameters associated with a specified model component.
It can use either data-driven priors (estimated from the subset of data corresponding to that component) or user-specified fixed priors.
By integrating these priors into the Bayesian inference framework, the function helps ensure that parameter estimates remain
grounded in reasonable values and incorporate domain knowledge or data-derived distributions.
}
\details{
This function supports two modes:
\enumerate{
  \item \strong{Data-Driven Priors:} When \code{use_data_priors = TRUE}, the function subsets the data for the specified \code{component},
        calculates summary statistics, and then draws from those statistics to form priors for each parameter.
        This approach allows the priors to adapt to the characteristics of the data associated with that component.

  \item \strong{User-Specified Fixed Priors:} When \code{use_data_priors = FALSE}, the function uses priors provided by the user
        through \code{user_fixed_priors}, bypassing any data-driven estimation.
}

In both modes, the priors are assumed to be normal distributions for each parameter. The function returns the sum of the log of these priors.
}
\examples{

# Example:
N <- 100
y <- rpois(N, lambda = 5)
x_vars <- list(
  list(runif(N)), # x1
  list(rnorm(N)), # x2
  list(rnorm(N)), # x3
  list(rnorm(N)) # x4
)
z <- sample(1:3, N, replace = TRUE)
params <- c(a = 1, b = 0.5, c = -0.2, d = 2, e = 0.3)

# Using data-driven priors for component 1
prior_val_data <- prior_combined(
  params = params,
  component = 1,
  y = y,
  x_vars = x_vars,
  z = z,
  use_data_priors = TRUE,
  user_fixed_priors = NULL
)

# Using user-specified fixed priors for component 2
#user_fixed_priors <- list(
#  component1 = list(
#    meany = 5, meanx1 = 0, meanx2 = 0, meanx3 = 0, meanx4 = 0,
#    sdy = 1, sdx1 = 1, sdx2 = 1, sdx3 = 1, sdx4 = 1
#  ),
#  component2 = list(
#    meany = 10, meanx1 = 1, meanx2 = 1, meanx3 = 1, meanx4 = 1,
#    sdy = 2, sdx1 = 2, sdx2 = 2, sdx3 = 2, sdx4 = 2
#  ),
#  component3 = list(
#    meany = 2, meanx1 = 2, meanx2 = 2, meanx3 = 2, meanx4 = 2,
#    sdy = 1, sdx1 = 1, sdx2 = 1, sdx3 = 1, sdx4 = 1
#  )
#)
#
#prior_val_fixed <- prior_combined(
#  params = params,
#  component = 2,
#  y = y,
#  x_vars = x_vars,
#  z = z,
#  use_data_priors = FALSE,
#  user_fixed_priors = user_fixed_priors
#)



}
