#' @title Proposal Function for the Potts Model Interaction Parameter
#'
#' @description
#' This function generates a proposal value for the interaction parameter in the Potts model,
#' drawn from a specified Beta distribution. It is typically used within the Markov Chain Monte Carlo (MCMC)
#' framework to propose new candidate values for the interaction parameter at each iteration.
#'
#' @usage
#' proposalfunction()
#'
#' @details
#' The Potts model describes configurations of states on a lattice, and the interaction parameter
#' influences how neighboring sites align. An MCMC algorithm typically requires a proposal distribution
#' to generate new candidate values for parameters. Here, the function draws from a \code{Beta(10,5)}
#' distribution, which, similar to the prior example, draws the proposals towards the higher end of the
#' \[0,1\] interval.
#'
#'
#' @return
#' A numeric value between 0 and 1, drawn from a Beta(10, 5) distribution, serving as a proposal value
#' for the interaction parameter in the Potts model.
#'
#' @examples
#' 
#' # Generate a proposal value for the Potts model interaction parameter
#' proposed_val <- proposalfunction()
#' #proposed_val
#' 
#'
#' @export
#'
proposalfunction <- function() {
  # draw proposal
  value <- rbeta(1, 10, 5)
  
  # quick sanity‐check
  stopifnot(
    is.numeric(value),
    length(value) == 1,
    is.finite(value),
    value >= 0,
    value <= 1
  )
  
  value
}

