% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base_methods.R
\name{base_coord}
\alias{base_coord}
\alias{base_methods}
\alias{base_coord,DNAStringSet_OR_NULL-method}
\alias{seq2granges}
\alias{seq2granges,DNAStringSet_OR_NULL-method}
\alias{base_seq2string_set}
\alias{base_seq2string_set,BaseSeq-method}
\alias{base_matrix}
\alias{base_matrix,DNAStringSet_OR_NULL-method}
\title{DNA Sequences Methods}
\usage{
base_coord(base = NULL, filepath = NULL, cube = "ACGT", group = "Z4", ...)

\S4method{base_coord}{DNAStringSet_OR_NULL}(
  base = NULL,
  filepath = NULL,
  cube = c("ACGT", "AGCT", "TCGA", "TGCA", "CATG", "GTAC", "CTAG", "GATC", "ACTG",
    "ATCG", "GTCA", "GCTA", "CAGT", "TAGC", "TGAC", "CGAT", "AGTC", "ATGC", "CGTA",
    "CTGA", "GACT", "GCAT", "TACG", "TCAG"),
  group = c("Z4", "Z5"),
  start = NA,
  end = NA,
  chr = 1L,
  strand = "+"
)

seq2granges(base = NULL, filepath = NULL, ...)

\S4method{seq2granges}{DNAStringSet_OR_NULL}(
  base = NULL,
  filepath = NULL,
  start = NA,
  end = NA,
  chr = 1L,
  strand = "+",
  seq_alias = NULL,
  ...
)

base_seq2string_set(x, ...)

\S4method{base_seq2string_set}{BaseSeq}(x)

base_matrix(base, ...)

\S4method{base_matrix}{DNAStringSet_OR_NULL}(
  base,
  cube = c("ACGT", "AGCT", "TCGA", "TGCA", "CATG", "GTAC", "CTAG", "GATC", "ACTG",
    "ATCG", "GTCA", "GCTA", "CAGT", "TAGC", "TGAC", "CGAT", "AGTC", "ATGC", "CGTA",
    "CTGA", "GACT", "GCAT", "TACG", "TCAG"),
  group = c("Z4", "Z5"),
  seq_alias = NULL
)
}
\arguments{
\item{base}{An object from a \code{\link[Biostrings]{DNAStringSet}} or
\code{\link[Biostrings]{DNAMultipleAlignment}} class carrying the DNA
pairwise alignment of two sequences.}

\item{filepath}{A character vector containing the path to a file in
\emph{\strong{fasta}} format to be read. This argument must be given if
\emph{codon & base} arguments are not provided.}

\item{cube}{A character string denoting one of the 24 Genetic-code cubes,
as given in references (2 2 3).}

\item{group}{A character string denoting the group representation for the
given base or codon as shown in reference (1).}

\item{...}{Not in use yet.}

\item{start, end, chr, strand}{Optional parameters required to build a
\code{\link[GenomicRanges]{GRanges-class}}. If not provided the default
values given for the function definition will be used.}

\item{seq_alias}{DNA sequence alias/ID and description.}

\item{x}{A 'BaseSeq' class object.}
}
\value{
Depending on the function called, different object will be
returned:
\subsection{Function 'base_coord'}{

This function returns a \code{\link{BaseGroup}} object
carrying the DNA sequence(s) and their respective coordinates in the
requested Abelian group of base representation (one-dimension, "Z4" or
"Z5"). Observe that to get coordinates in the set of of integer numbers
("Z") is also possible but they are not defined to integrate a Abelian
group. These are just used for the further insertion the codon set in the
3D space (R^3).
}

\subsection{Function 'seq2granges'}{

This function returns a \link{BaseGroup} object carrying the DNA sequence(s), one
base per ranges. A \link{BaseGroup} class object inherits from
\code{\link[GenomicRanges]{GRanges-class}}.
}

\subsection{Function 'base_seq2string_set'}{

This function returns a \code{\link[Biostrings]{DNAStringSet-class}}.
}

A BaseGroup-class object.
}
\description{
\subsection{Base coordinates on a given Abelian group representation}{

Given a string denoting a codon or base from the DNA (or RNA) alphabet,
function \strong{\emph{base_coord}} return the base coordinates in the
specify genetic-code Abelian group, as given in reference (1).
}

\subsection{DNA sequences to \code{\link[GenomicRanges]{GRanges}} of bases.}{

Function \strong{\emph{seq2granges}} transform an object from
\code{\link[Biostrings]{DNAStringSet}},
\code{\link[Biostrings]{DNAMultipleAlignment-class}} or a character into
an object from \link{BaseSeq}.
}

\subsection{BaseSeq-class object to DNAStringSet-class object.}{

Function \strong{\emph{base_seq2string_set}} transforms an object from
\link{BaseSeq} into an object from \code{\link[Biostrings]{DNAStringSet-class}}.
}
}
\details{
\subsection{Function 'base_coord'}{

Function \strong{\emph{base_coord}} is defined only for pairwise
aligned sequences. Symbols "-" and "N" usually found in DNA sequence
alignments to denote gaps and missing/unknown bases are represented by the
number: '-1' on Z4 and '0' on Z5. In Z64 the symbol 'NA' will be returned
for codons including symbols "-" and "N".
}

\subsection{Functions 'seq2granges' and 'base_seq2string_set'}{

For the sake of brevity the metacolumns from the object returned by
function 'seq2granges' are named as 'S1', 'S2', 'S3', and so on. The
original DNA sequence alias are stored in the slot named 'seq_alias'.
(see examples).
}
}
\examples{
## Example 1. Let's get the base coordinates for codons "ACG"
## and "TGC":
x0 <- c("ACG", "TGC")
x1 <- DNAStringSet(x0)
x1

## Get the base coordinates on cube = "ACGT" on the Abelian group = "Z4"
base_coord(x1, cube = "ACGT", group = "Z4")

## Example 2. Load a pairwise alignment
data("aln", package = "GenomAutomorphism")
aln

## DNA base representation in the Abelian group Z4
bs_cor <- base_coord(
    base = aln,
    cube = "ACGT"
)
bs_cor

## Example 3. DNA base representation in the Abelian group Z5
bs_cor <- base_coord(
    base = aln,
    cube = "ACGT",
    group = "Z5"
)
bs_cor

## Example 4. Load a multiple sequence alignment (MSA) of primate BRCA1 DNA  
## repair genes 
data("brca1_aln2", package = "GenomAutomorphism")
brca1_aln2

## Get BaseSeq-class object
gr <- seq2granges(brca1_aln2)
gr

## Transform the BaseSeq-class object into a DNAStringSet-class object
str_set <- base_seq2string_set(gr)
str_set

## Recovering the original MSA
DNAMultipleAlignment(as.character(str_set))

## Example 5. 
base_matrix(base = aln, cube = "CGTA", group = "Z5")

## Example 5. 

}
\references{
\enumerate{
\item Robersy Sanchez, Jesus Barreto (2021) Genomic Abelian Finite
Groups.
\href{https://doi.org/10.1101/2021.06.01.446543}{doi:10.1101/2021.06.01.446543}
\item M. V Jose, E.R. Morgado, R. Sanchez, T. Govezensky, The 24 possible
algebraic representations of the standard genetic code in six or in three
dimensions, Adv. Stud. Biol. 4 (2012) 119-152.\href{https://is.gd/na9eap}{PDF}.
\item R. Sanchez. Symmetric Group of the Genetic-Code Cubes. Effect of the
Genetic-Code Architecture on the Evolutionary Process MATCH Commun. Math.
Comput. Chem. 79 (2018) 527-560.
}
}
\seealso{
\href{https://github.com/genomaths/GenomeAlgebra_SymmetricGroup}{Symmetric Group of the Genetic-Code Cubes.}

\code{\link{codon_coord}} and \code{\link{base2int}}.

\href{https://github.com/genomaths/GenomeAlgebra_SymmetricGroup}{Symmetric Group of the Genetic-Code Cubes.}

\link{base_coord} and \link{codon_coord}.
}
\author{
Robersy Sanchez \url{https://genomaths.com}
}
