#' Pipeline for bulk long read RNA-seq data processing
#'
#' @md
#'
#' @description Semi-supervised isofrom detection and annotation for long read data.
#' This variant is meant for bulk samples. Specific parameters can be configured in
#' the config file (see \code{\link{create_config}}), input files are specified via
#' arguments.
#'
#' @details
#' By default FLAMES use minimap2 for read alignment. After the genome alignment step (\code{do_genome_align}), FLAMES summarizes the alignment for each read by grouping reads
#' with similar splice junctions to get a raw isoform annotation (\code{do_isoform_id}). The raw isoform
#' annotation is compared against the reference annotation to correct potential splice site
#' and transcript start/end errors. Transcripts that have similar splice junctions
#' and transcript start/end to the reference transcript are merged with the
#' reference. This process will also collapse isoforms that are likely to be truncated
#' transcripts. If \code{isoform_id_bambu} is set to \code{TRUE}, \code{bambu::bambu} will be used to generate the updated annotations.
#' Next is the read realignment step (\code{do_read_realign}), where the sequence of each transcript from the update annotation is extracted, and
#' the reads are realigned to this updated \code{transcript_assembly.fa} by minimap2. The
#' transcripts with only a few full-length aligned reads are discarded.
#' The reads are assigned to transcripts based on both alignment score, fractions of
#' reads aligned and transcript coverage. Reads that cannot be uniquely assigned to
#' transcripts or have low transcript coverage are discarded. The UMI transcript
#' count matrix is generated by collapsing the reads with the same UMI in a similar
#' way to what is done for short-read scRNA-seq data, but allowing for an edit distance
#' of up to 2 by default. Most of the parameters, such as the minimal distance to splice site and minimal percentage of transcript coverage
#' can be modified by the JSON configuration file (\code{config_file}).
#'
#' @param config_file Path to the JSON configuration file. See \code{\link{create_config}} for creating one.
#' @param outdir Path to the output directory. If it does not exist, it will be created.
#' @param fastq Path to the FASTQ file or a directory containing FASTQ files. Each file
#'   will be processed as an individual sample.
#' @param annotation The file path to the annotation file in GFF3 / GTF format.
#' @param genome_fa The file path to the reference genome in FASTA format.
#' @param genome_mmi (optional) The file path to minimap2's index reference genome.
#' @param minimap2 (optional) The path to the minimap2 binary. If not provided, FLAMES will
#'   use a copy from bioconda via \code{basilisk}.
#' @param samtools (optional) The path to the samtools binary. If not provided, FLAMES will
#'   use a copy from bioconda via \code{basilisk}.
#' @param controllers (optional, **experimental**) A \code{crew_class_controller} object for running certain steps
#' @return A \code{FLAMES.Pipeline} object. The pipeline could be run using \code{\link{run_FLAMES}}, and / or resumed using \code{\link{resume_FLAMES}}.
#'
#' @seealso
#' \code{\link{create_config}} for creating a configuration file,
#' \code{\link{SingleCellPipeline}} for single cell pipelines,
#' \code{\link{MultiSampleSCPipeline}} for multi sample single cell pipelines.
#'
#' @importFrom utils file_test
#'
#' @examples
#' outdir <- tempfile()
#' dir.create(outdir)
#' # simulate 3 samples via sampling
#' reads <- ShortRead::readFastq(
#'   system.file("extdata", "fastq", "musc_rps24.fastq.gz", package = "FLAMES")
#' )
#' dir.create(file.path(outdir, "fastq"))
#' ShortRead::writeFastq(reads[1:100],
#'   file.path(outdir, "fastq/sample1.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' reads <- reads[-(1:100)]
#' ShortRead::writeFastq(reads[1:100],
#'   file.path(outdir, "fastq/sample2.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' reads <- reads[-(1:100)]
#' ShortRead::writeFastq(reads,
#'   file.path(outdir, "fastq/sample3.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' # prepare the reference genome
#' genome_fa <- file.path(outdir, "rps24.fa")
#' R.utils::gunzip(
#'   filename = system.file("extdata", "rps24.fa.gz", package = "FLAMES"),
#'   destname = genome_fa, remove = FALSE
#' )
#' ppl <- BulkPipeline(
#'   fastq = c(
#'     "sample1" = file.path(outdir, "fastq", "sample1.fq.gz"),
#'     "sample2" = file.path(outdir, "fastq", "sample2.fq.gz"),
#'     "sample3" = file.path(outdir, "fastq", "sample3.fq.gz")
#'   ),
#'   annotation = system.file("extdata", "rps24.gtf.gz", package = "FLAMES"),
#'   genome_fa = genome_fa,
#'   config_file = create_config(outdir, type = "sc_3end", threads = 1, no_flank = TRUE),
#'   outdir = outdir
#' )
#' ppl <- run_FLAMES(ppl) # run the pipeline
#' experiment(ppl) # get the result as SummarizedExperiment
#'
#' @export
BulkPipeline <- function(
    config_file, outdir, fastq, annotation, genome_fa, genome_mmi,
    minimap2, samtools, controllers) {
  pipeline <- new("FLAMES.Pipeline")
  config <- check_arguments(annotation, fastq, genome_bam = NULL, outdir, genome_fa, config_file)$config

  if (!dir.exists(outdir)) {
    dir.create(outdir)
    message(sprintf("Output directory (%s) did not exist, created.", outdir))
  }

  if (length(fastq) == 1 && utils::file_test("-d", fastq)) {
    fastq <- list.files(fastq, pattern = "\\.(fastq|fq)(\\.gz)?$", full.names = TRUE)
  } else if (!all(utils::file_test("-f", fastq))) {
    stop("fastq must be a valid path to a folder or a FASTQ file")
  }
  if (is.null(names(fastq))) {
    names(fastq) <- gsub("\\.(fastq|fq)(\\.gz)?$", "", basename(fastq))
  }
  names(fastq) <- make.unique(names(fastq), sep = "_")

  steps <- c(
    "genome_alignment", "isoform_identification",
    "read_realignment", "transcript_quantification"
  )
  steps <- config$pipeline_parameters[paste0("do_", steps)] |>
    unlist() |>
    setNames(steps)
  message("Configured steps: \n", paste0("\t", names(steps), ": ", steps, collapse = "\n"))


  # assign slots
  ## inputs
  pipeline@config <- config
  pipeline@outdir <- outdir
  pipeline@fastq <- fastq
  pipeline@annotation <- annotation
  pipeline@genome_fa <- genome_fa
  if (!missing(genome_mmi) && is.character(genome_mmi)) {
    pipeline@genome_mmi <- genome_mmi
  }

  ## outputs
  # metadata
  pipeline@bed <- file.path(outdir, "reference.bed")
  pipeline@genome_bam <- file.path(outdir, paste0(names(fastq), "_", "align2genome.bam"))
  pipeline@transcriptome_bam <- file.path(outdir, paste0(names(fastq), "_", "realign2transcript.bam"))
  pipeline@transcriptome_assembly <- file.path(outdir, "transcript_assembly.fa")
  pipeline@experiment <- file.path(outdir, "experiment.rds")

  ## binaries
  if (missing(minimap2) || !is.character(minimap2)) {
    minimap2 <- find_bin("minimap2")
    if (is.na(minimap2)) {
      stop("minimap2 not found, please make sure it is installed and provide its path as the minimap2 argument")
    }
  }
  pipeline@minimap2 <- minimap2
  if (missing(samtools) || !is.character(samtools)) {
    samtools <- find_bin("samtools")
  }
  if (is.na(samtools)) {
    message("samtools not found, will use Rsamtools package instead")
  }
  pipeline@samtools <- samtools
  ##

  ## pipeline state
  pipeline@steps <- steps
  pipeline@completed_steps <- setNames(
    rep(FALSE, length(steps)), names(steps)
  )

  if (!missing(controllers)) {
    pipeline@controllers <- normalize_controllers(controllers, names(steps))
  }

  # TODO: add resume option
  # validate if e.g. genome_bam exists, skip genome alignment step

  return(pipeline)
}

normalize_controllers <- function(controllers, step_names) {
  if (inherits(controllers, "crew_class_controller")) {
    # Use the same controller for all steps
    list(default = controllers)
  } else if (is.list(controllers) && all(sapply(controllers, inherits, "crew_class_controller"))) {
    controllers
  } else {
    stop("`controllers` must be a crew_class_controller or a named list of crew_class_controller objects.")
  }
}

setGeneric("prerun_check", function(pipeline, overwrite = FALSE) {
  standardGeneric("prerun_check")
})
setMethod("prerun_check", "FLAMES.Pipeline", function(pipeline, overwrite = FALSE) {
  # return TRUE for run_FLAMES to proceed
  # return FALSE for run_FLAMES to stop gracefully, when all steps are completed
  # stop() when pipeline is partially completed and user does not want to overwrite
  if (all(pipeline@completed_steps)) {
    message("All steps have already been completed.")
    if (overwrite) {
      warning("Re-running pipeline to overwrite existing results.")
      return(TRUE)
    } else {
      message("Pipeline is already completed. Set overwrite = TRUE to re-run.")
      return(FALSE)
    }
  } else if (any(pipeline@completed_steps)) {
    message("Some steps have already been completed.")
    if (overwrite) {
      warning("Re-running pipeline to overwrite existing results.")
      return(TRUE)
    } else {
      # TODO: implement resuming and prompt user to resume
      stop("Pipeline is partially completed. Please set overwrite = TRUE to proceed.")
    }
  } else {
    # nothing done yet, proceed
    return(TRUE)
  }
})

#' Execute a single step of the FLAMES pipeline
#'
#' @description This function runs the specified step of the FLAMES pipeline.
#'
#' @param pipeline A FLAMES.Pipeline object.
#' @param step The step to run. One of "barcode_demultiplex", "genome_alignment",
#'   "gene_quantification", "isoform_identification", "read_realignment", or
#'  "transcript_quantification".
#' @param disable_controller (optional) If TRUE, the step will be executed in
#' the current R session, instead of using crew controllers.
#' @return An updated FLAMES.Pipeline object.
#'
#' @seealso
#' \code{\link{run_FLAMES}} to run the entire pipeline.
#' \code{\link{resume_FLAMES}} to resume a pipeline from the last completed step.
#'
#' @examples
#' pipeline <- example_pipeline("BulkPipeline")
#' pipeline <- run_step(pipeline, "genome_alignment")
#'
#' @export
setGeneric("run_step", function(pipeline, step, disable_controller = TRUE) {
  standardGeneric("run_step")
})
#' @importFrom cli cli_rule
#' @rdname run_step
#' @export
setMethod("run_step", "FLAMES.Pipeline", function(pipeline, step, disable_controller = TRUE) {

  start_time <- Sys.time()
  cli::cli_rule(sprintf("Running step: %s @ %s", step, date()))
  controllers <- pipeline@controllers
  if (disable_controller) {
    pipeline@controllers <- list()
  }

  controller_handling_steps <- c(
    # these steps handles controllers internally
    "barcode_demultiplex",
    "genome_alignment",
    "read_realignment"
  )

  if (!any(c(step, "default") %in% names(pipeline@controllers)) ||
        step %in% controller_handling_steps) {
    pipeline <- switch(step,
      barcode_demultiplex = barcode_demultiplex(pipeline),
      genome_alignment = genome_alignment(pipeline),
      gene_quantification = gene_quantification(pipeline),
      isoform_identification = isoform_identification(pipeline),
      read_realignment = read_realignment(pipeline),
      transcript_quantification = transcript_quantification(pipeline),
      stop(sprintf("Unknown step: %s", step))
    )
  } else {
    controller <- if (step %in% names(pipeline@controllers)) {
      pipeline@controllers[[step]]
    } else if ("default" %in% names(pipeline@controllers)) {
      pipeline@controllers[["default"]]
    } else {
      stop("Unexpected error: no controller found for step ", step)
    }
    if (controller$started()) {
      tryCatch({
        controller$terminate()
      }, error = function(e) {
        warning(sprintf("Error terminating controller: %s", e$message))
      })
    }
    controller$start()
    controller$push(
      command =
        switch(step,
          # barcode_demultiplex = FLAMES:::barcode_demultiplex(pipeline),
          # genome_alignment = FLAMES:::genome_alignment(pipeline),
          gene_quantification = FLAMES:::gene_quantification(pipeline),
          isoform_identification = FLAMES:::isoform_identification(pipeline),
          # read_realignment = FLAMES:::read_realignment(pipeline),
          transcript_quantification = FLAMES:::transcript_quantification(pipeline),
          stop(sprintf("Unknown / unexpected step: %s", step))
        ),
      data = list(pipeline = pipeline, step = step),
    )
    controller$wait(mode = "all")
    task <- controller$pop(error = "stop")
    pipeline <- task$result[[1]]
    controller$terminate()
  }

  end_time <- Sys.time()
  pipeline@completed_steps[step] <- TRUE
  # clear last error if it was successfully completed
  if (length(pipeline@last_error) > 0 && pipeline@last_error$step == step) {
    pipeline@last_error <- list()
  }
  pipeline@durations[step] <- difftime(end_time, start_time, units = "secs")
  if (disable_controller) {
    pipeline@controllers <- controllers
  }

  return(pipeline)
})

#' Execute a FLAMES pipeline
#'
#' @description This function runs the FLAMES pipeline. It will run all steps in the pipeline.
#' @param pipeline A FLAMES.Pipeline object.
#' @param overwrite (optional) If TRUE, the pipeline will be re-run even if some steps are already completed.
#' @return An updated FLAMES.Pipeline object.
#' @seealso
#' \code{\link{resume_FLAMES}} to resume a pipeline from the last completed step.
#' @examples
#' pipeline <- example_pipeline("BulkPipeline")
#' pipeline <- run_FLAMES(pipeline)
#' @export
setGeneric("run_FLAMES", function(pipeline, overwrite = FALSE) {
  standardGeneric("run_FLAMES")
})
#' @rdname run_FLAMES
#' @export
setMethod("run_FLAMES", "FLAMES.Pipeline", function(pipeline, overwrite = FALSE) {
  if (!prerun_check(pipeline, overwrite)) {
    return(pipeline)
  }

  for (step in names(which(pipeline@steps))) {
    # S4 objects are immutable
    # Need R6 for passing by reference
    pipeline <- tryCatch(
      run_step(pipeline, step, disable_controller = FALSE),
      error = function(e) {
        warning(sprintf("Error in step %s: %s, pipeline stopped.", step, e$message))
        pipeline@last_error <- list(
          step = step,
          error = e,
          traceback = capture.output(traceback())
        )
        return(pipeline)
      }
    )
    if (!pipeline@completed_steps[step]) {
      break
    }
  }
  return(pipeline)
})

#' Resume a FLAMES pipeline
#'
#' @description This function resumes a FLAMES pipeline by running configured
#' but unfinished steps.
#' @param pipeline A FLAMES.Pipeline object.
#' @return An updated FLAMES.Pipeline object.
#' @seealso
#' \code{\link{run_FLAMES}} to run the entire pipeline.
#' @examples
#' pipeline <- example_pipeline("BulkPipeline")
#' pipeline <- run_step(pipeline, "genome_alignment")
#' pipeline <- resume_FLAMES(pipeline)
#' @export
setGeneric("resume_FLAMES", function(pipeline) {
  standardGeneric("resume_FLAMES")
})
#' @rdname resume_FLAMES
#' @export
setMethod("resume_FLAMES", "FLAMES.Pipeline", function(pipeline) {
  configured_steps <- pipeline@completed_steps[pipeline@steps]
  unfinished_steps <- names(which(!configured_steps))
  if (length(unfinished_steps) == 0) {
    message("All steps have already been completed.")
    return(pipeline)
  } else {
    message("Resuming pipeline from step: ", unfinished_steps[1])
    for (step in unfinished_steps) {
      pipeline <- tryCatch(
        run_step(pipeline, step, disable_controller = FALSE),
        error = function(e) {
          warning(sprintf("Error in step %s: %s, pipeline stopped.", step, e$message))
          pipeline@last_error <- list(
            step = step,
            error = e,
            traceback = capture.output(traceback())
          )
          return(pipeline)
        }
      )
      if (!pipeline@completed_steps[step]) {
        break
      }
    }
    return(pipeline)
  }
})

# Getters and setters

#' Steps to perform in the pipeline
#'
#' @param pipeline An object of class `FLAMES.Pipeline`
#' @return A named logical vector containing all possible steps
#' for the pipeline. The names of the vector are the step names,
#' and the values are logical indicating whether the step is
#' configured to be performed.
#' @examples
#' ppl <- example_pipeline()
#' steps(ppl)
#' @export
setGeneric("steps", function(pipeline) standardGeneric("steps"))
#' @rdname steps
#' @export
setMethod("steps", "FLAMES.Pipeline", function(pipeline) {
  pipeline@steps
})

#' Set steps to perform in the pipeline
#'
#' @param pipeline An object of class `FLAMES.Pipeline`
#' @param value A named logical vector containing all possible steps
#' for the pipeline. The names of the vector are the step names,
#' and the values are logical indicating whether the step is
#' configured to be performed.
#' @return An pipeline of class `FLAMES.Pipeline` with the updated steps.
#' @examples
#' ppl <- example_pipeline()
#' steps(ppl) <- c(
#'   barcode_demultiplex = TRUE,
#'   genome_alignment = TRUE,
#'   gene_quantification = TRUE,
#'   isoform_identification = FALSE,
#'   read_realignment = FALSE,
#'   transcript_quantification = TRUE
#' )
#' ppl
#' # or partially change a step:
#' steps(ppl)["read_realignment"] <- TRUE
#' ppl
#' @export
setGeneric("steps<-", function(pipeline, value) standardGeneric("steps<-"))
#' @rdname steps-set
#' @export
setMethod("steps<-", "FLAMES.Pipeline", function(pipeline, value) {
  # validate the names
  if (any(is.null(names(value))) || any(is.na(names(value)))) {
    stop("Steps must be a named logical vector.")
  }
  if (!all(names(value) %in% names(pipeline@steps))) {
    stop(sprintf(
      "Invalid step names. Expected: %s, but got: %s",
      paste(names(pipeline@steps), collapse = ", "),
      paste(names(value)[!names(value) %in% names(pipeline@steps)], collapse = ", ")
    ))
  }
  pipeline@steps[names(value)] <- value
  pipeline
})

#' Get pipeline configurations
#'
#' @description This function returns the configuration of the pipeline.
#' @param pipeline An object of class `FLAMES.Pipeline`.
#' @return A list containing the configuration of the pipeline.
#' @examples
#' pipeline <- example_pipeline(type = "BulkPipeline")
#' config(pipeline)
#' @export
setGeneric("config", function(pipeline) standardGeneric("config"))
#' @rdname config
#' @export
setMethod("config", "FLAMES.Pipeline", function(pipeline) {
  pipeline@config
})

#' Set pipeline configurations
#'
#' @description This function sets the configuration of the pipeline.
#' @param pipeline An pipeline of class `FLAMES.Pipeline`.
#' @param value A list containing the configuration of the pipeline, or
#' a path to a JSON configuration file.
#' @return An pipeline of class `FLAMES.Pipeline` with the updated configuration.
#' @examples
#' pipeline <- example_pipeline(type = "BulkPipeline")
#' # Set a new configuration
#' config(pipeline) <- create_config(outdir = tempdir())
#' @export
setGeneric("config<-", function(pipeline, value) standardGeneric("config<-"))
#' @rdname config-set
#' @export
setMethod("config<-", "FLAMES.Pipeline", function(pipeline, value) {
  if (is.character(value) && file.exists(value)) {
    value <- load_config(value)
  } else if (!is.list(value)) {
    stop("Configuration must be a list or a path to a JSON configuration file.")
  }
  pipeline@config <- value
  pipeline
})

#' Get controllers
#'
#' @description Gets the controllers for the pipeline.
#' @param pipeline A FLAMES.Pipeline object.
#' @return A named list of \code{crew_class_controller} objects, where each
#' controller corresponds to a step in the pipeline.
#' @examples
#' pipeline <- example_pipeline(type = "MultiSampleSCPipeline")
#' controllers(pipeline) # get the controllers
#' @export
setGeneric("controllers", function(pipeline) {
  standardGeneric("controllers")
})
#' @rdname controllers
#' @export
setMethod("controllers", "FLAMES.Pipeline", function(pipeline) {
  pipeline@controllers
})
#' Set controllers
#'
#' @description Sets the controllers for the pipeline.
#' @param pipeline A FLAMES.Pipeline object.
#' @param value A \code{crew_class_controller} object or a named list of
#' \code{crew_class_controller} objects. If a single controller is provided,
#' it will be used for all steps in the pipeline. If a named list is provided,
#' steps with names that match the names of the list will use the corresponding
#' controller, and steps without a specified controller will use the current R
#' session.
#' @return An updated FLAMES.Pipeline object with the specified controllers.
#' @examples
#' pipeline <- example_pipeline()
#' # Only set the genome alignment controller
#' controllers(pipeline) <- list(genome_alignment = crew::crew_controller_local())
#' # Same as above
#' controllers(pipeline)[["genome_alignment"]] <- crew::crew_controller_local()
#' # Set a controller for all steps
#' controllers(pipeline) <- crew::crew_controller_local()
#' # Unset all controllers and use the current R session
#' controllers(pipeline) <- list()
#' @export
setGeneric("controllers<-", function(pipeline, value) {
  standardGeneric("controllers<-")
})
#' @rdname controllers-set
#' @export
setMethod("controllers<-", "FLAMES.Pipeline", function(pipeline, value) {
  pipeline@controllers <- normalize_controllers(value, names(pipeline@steps))
  pipeline
})

#' Get pipeline results
#'
#' @description This function returns the results of the pipeline as a
#' \code{SummarizedExperiment} object, a \code{SingleCellExperiment} object, or a
#' list of \code{SingleCellExperiment} objects, depending on the pipeline type.
#' @param pipeline A FLAMES.Pipeline object.
#' @return A \code{SummarizedExperiment} object, a \code{SingleCellExperiment} object,
#' or a list of \code{SingleCellExperiment} objects.
#' @examples
#' pipeline <- example_pipeline(type = "BulkPipeline")
#' pipeline <- run_FLAMES(pipeline)
#' se <- experiment(pipeline)
#' @export
setGeneric("experiment", function(pipeline) {
  standardGeneric("experiment")
})
#' @rdname experiment
#' @export
setMethod("experiment", "FLAMES.Pipeline", function(pipeline) {
  if (is.na(pipeline@experiment)) {
    return(NULL)
  }
  if (file.exists(pipeline@experiment)) {
    return(readRDS(pipeline@experiment))
  } else {
    warning(sprintf("Experiment file not found: %s", pipeline@experiment))
    return(NULL)
  }
})

# individual steps as methods
# dummy methods
setGeneric("barcode_demultiplex", function(pipeline) {
  standardGeneric("barcode_demultiplex")
})
setMethod("barcode_demultiplex", "FLAMES.Pipeline", function(pipeline) {
  stop("Barcode demultiplexing is only implemented for single cell pipelines (SingleCellPipeline() and MultiSampleSCPipeline())")
})
setGeneric("gene_quantification", function(pipeline) {
  standardGeneric("gene_quantification")
})
setMethod("gene_quantification", "FLAMES.Pipeline", function(pipeline) {
  # todo: implement gene quantification for bulk pipelines
  stop("Gene quantification is not implemented for bulk pipelines yet")
})

setGeneric("genome_alignment_raw", function(pipeline, fastqs, include_tags = FALSE) {
  standardGeneric("genome_alignment_raw")
})
setMethod("genome_alignment_raw", "FLAMES.Pipeline", function(pipeline, fastqs, include_tags = FALSE) {
  minimap2_args <- c(
    "-ax", "splice", "-k14", "--secondary=no", # "-y",
    "-t", pipeline@config$pipeline_parameters$threads,
    "--seed", pipeline@config$pipeline_parameters$seed
  )
  if (include_tags) {
    minimap2_args <- base::append(minimap2_args, "-y")
  }
  if (pipeline@config$alignment_parameters$no_flank) {
    minimap2_args <- base::append(minimap2_args, "--splice-flank=no")
  }

  # replace k8 paftools.js gff2bed gff > bed12 with rtracklayer::export.bed
  if (pipeline@config$alignment_parameters$use_junctions) {
    if (is.na(pipeline@bed)) {
      pipeline@bed <- file.path(pipeline@outdir, "reference.bed")
    }
    if (!file.exists(pipeline@bed)) {
      message("Creating junction bed file from GFF3 annotation.")
      gff2bed(gff = pipeline@annotation, bed = pipeline@bed)
    }
    minimap2_args <- base::append(
      minimap2_args,
      c("--junc-bed", pipeline@bed, "--junc-bonus", "1")
    )
  }

  # use genome_mmi if provided
  if (!is.na(pipeline@genome_mmi) && file.exists(pipeline@genome_mmi)) {
    genome <- pipeline@genome_mmi
  } else {
    genome <- pipeline@genome_fa
  }


  samples <- if (!is.null(names(fastqs))) names(fastqs) else fastqs
  if (!any(c("genome_alignment", "default") %in% names(pipeline@controllers))) {
    res <- lapply(
      seq_along(fastqs),
      function(i) {
        message(sprintf("Aligning sample %s -> %s", samples[i], pipeline@genome_bam[i]))
        minimap2_align(
          fq_in = fastqs[i],
          fa_file = genome,
          config = pipeline@config,
          outfile = pipeline@genome_bam[i],
          minimap2_args = minimap2_args,
          sort_by = "coordinates",
          minimap2 = pipeline@minimap2,
          samtools = pipeline@samtools,
          threads = pipeline@config$pipeline_parameters$threads,
          tmpdir = pipeline@outdir
        )
      }
    )
  } else {
    controller <- if ("genome_alignment" %in% names(pipeline@controllers)) {
      pipeline@controllers[["genome_alignment"]]
    } else if ("default" %in% names(pipeline@controllers)) {
      pipeline@controllers[["default"]]
    } else {
      stop("Unexpected error: no controller found for genome alignment step.")
    }
    if (controller$started()) {
      tryCatch({
        controller$terminate()
      }, error = function(e) {
        warning(sprintf("Error terminating controller: %s", e$message))
      })
    }
    controller$start()
    crew_result <- controller$map(
      command = {
        minimap2_align(
          fq_in = fastqs[j],
          fa_file = genome,
          config = config,
          outfile = genome_bam[j],
          minimap2_args = minimap2_args,
          sort_by = "coordinates",
          minimap2 = minimap2,
          samtools = samtools,
          threads = threads,
          tmpdir = outdir
        )
      },
      iterate = list(j = seq_along(fastqs)),
      data = list(
        minimap2_align = FLAMES:::minimap2_align,
        samples = samples,
        fastqs = fastqs,
        genome_bam = pipeline@genome_bam,
        genome = genome,
        config = pipeline@config,
        minimap2_args = minimap2_args,
        minimap2 = pipeline@minimap2,
        samtools = pipeline@samtools,
        threads = pipeline@config$pipeline_parameters$threads,
        outdir = pipeline@outdir
      ),
      error = "stop"
    )
    controller$terminate()
    res <- crew_result$result
    message("Alignment complete for the following samples:")
    message(
      paste0(samples, " ->", pipeline@genome_bam, collapse = "\n")
    )
  }

  if (!is.null(names(fastqs))) {
    names(res) <- names(fastqs)
  }
  pipeline@metadata$genome_alignment <- res
  return(pipeline)
})
setGeneric("genome_alignment", function(pipeline) {
  standardGeneric("genome_alignment")
})
setMethod("genome_alignment", "FLAMES.Pipeline", function(pipeline) {
  genome_alignment_raw(
    pipeline = pipeline,
    fastqs = pipeline@fastq
  )
})

setGeneric("isoform_identification", function(pipeline) {
  standardGeneric("isoform_identification")
})
setMethod("isoform_identification", "FLAMES.Pipeline", function(pipeline) {
  if (pipeline@config$pipeline_parameters$bambu_isoform_identification) {
    novel_isoform_annotation <- find_isoform_bambu(
      annotation = pipeline@annotation,
      genome_fa = pipeline@genome_fa,
      genome_bam = pipeline@genome_bam,
      outdir = pipeline@outdir,
      config = pipeline@config
    )
  } else {
    novel_isoform_annotation <- find_isoform_flames(
      annotation = pipeline@annotation,
      genome_fa = pipeline@genome_fa,
      genome_bam = pipeline@genome_bam,
      outdir = pipeline@outdir,
      config = pipeline@config
    )
  }
  pipeline@novel_isoform_annotation <- novel_isoform_annotation
  annotation_to_fasta(
    isoform_annotation = novel_isoform_annotation,
    genome_fa = pipeline@genome_fa,
    outfile = pipeline@transcriptome_assembly
  )
  return(pipeline)
})

setGeneric("read_realignment_raw", function(pipeline, include_tags = FALSE, sort_by, fastqs) {
  standardGeneric("read_realignment_raw")
})
setMethod(
  "read_realignment_raw", "FLAMES.Pipeline",
  function(pipeline, include_tags = FALSE, sort_by, fastqs) {
    if (pipeline@config$pipeline_parameters$oarfish_quantification) {
      minimap2_args <- c(
        "--eqx", "-N", "100", "-ax", "map-ont",
        "-t", pipeline@config$pipeline_parameters$threads,
        "--seed", pipeline@config$pipeline_parameters$seed
      )
    } else {
      minimap2_args <- c(
        "-ax", "map-ont", "-p", "0.9", "--end-bonus", "10", "-N", "3",
        "-t", pipeline@config$pipeline_parameters$threads,
        "--seed", pipeline@config$pipeline_parameters$seed
      )
    }
    if (include_tags) {
      minimap2_args <- base::append(minimap2_args, "-y")
    }

    if (!file.exists(pipeline@transcriptome_assembly)) {
      if (!pipeline@steps["isoform_identification"]) {
        message("Using reference annotation for transcriptome assembly.")
        annotation_to_fasta(
          isoform_annotation = pipeline@annotation,
          genome_fa = pipeline@genome_fa,
          outfile = pipeline@transcriptome_assembly
        )
      } else {
        stop("Isoform identification step configured but transcriptome assembly file does not exist, aborting realignment.")
      }
    }

    samples <- if (!is.null(names(fastqs))) names(fastqs) else fastqs
    if (!any(c("read_realignment", "default") %in% names(pipeline@controllers))) {
      res <- lapply(
        seq_along(fastqs),
        function(i) {
          message(sprintf(
            "Realigning sample %s -> %s",
            samples[i], pipeline@transcriptome_bam[i]
          ))
          minimap2_align(
            fq_in = fastqs[i],
            fa_file = pipeline@transcriptome_assembly,
            config = pipeline@config,
            outfile = pipeline@transcriptome_bam[i],
            minimap2_args = minimap2_args,
            sort_by = sort_by,
            minimap2 = pipeline@minimap2,
            samtools = pipeline@samtools,
            threads = pipeline@config$pipeline_parameters$threads,
            tmpdir = pipeline@outdir
          )
        }
      )
    } else {
      controller <- if ("read_realignment" %in% names(pipeline@controllers)) {
        pipeline@controllers[["read_realignment"]]
      } else if ("default" %in% names(pipeline@controllers)) {
        pipeline@controllers[["default"]]
      } else {
        stop("Unexpected error: no controller found for read realignment step.")
      }
      if (controller$started()) {
        tryCatch({
          controller$terminate()
        }, error = function(e) {
          warning(sprintf("Error terminating controller: %s", e$message))
        })
      }
      controller$start()
      crew_result <- controller$map(
        command = {
          minimap2_align(
            fq_in = fastqs[j],
            fa_file = transcriptome_assembly,
            config = config,
            outfile = transcriptome_bam[j],
            minimap2_args = minimap2_args,
            sort_by = sort_by,
            minimap2 = minimap2,
            samtools = samtools,
            threads = config$pipeline_parameters$threads,
            tmpdir = outdir
          )
        },
        iterate = list(j = seq_along(fastqs)),
        data = list(
          minimap2_align = FLAMES:::minimap2_align,
          samples = samples,
          fastqs = fastqs,
          transcriptome_assembly = pipeline@transcriptome_assembly,
          config = pipeline@config,
          transcriptome_bam = pipeline@transcriptome_bam,
          minimap2_args = minimap2_args,
          sort_by = sort_by,
          minimap2 = pipeline@minimap2,
          samtools = pipeline@samtools,
          outdir = pipeline@outdir
        )
      )
      controller$terminate()
      res <- crew_result$result
      message("Realignment complete for the following samples:")
      message(
        paste0(samples, " ->", pipeline@transcriptome_bam, collapse = "\n")
      )
    }

    if (!is.null(names(fastqs))) {
      names(res) <- names(fastqs)
    }
    pipeline@metadata$read_realignment <- res
    return(pipeline)
  }
)

setGeneric("read_realignment", function(pipeline, include_tags = FALSE) {
  standardGeneric("read_realignment")
})
setMethod("read_realignment", "FLAMES.Pipeline", function(pipeline, include_tags = FALSE) {
  sort_by <- ifelse(
    pipeline@config$pipeline_parameters$oarfish_quantification,
    "none",
    "coordinates"
  )
  read_realignment_raw(
    pipeline = pipeline,
    include_tags = include_tags,
    sort_by = sort_by,
    fastqs = pipeline@fastq
  )
})
setGeneric("transcript_quantification", function(pipeline, reference_only) {
  standardGeneric("transcript_quantification")
})
setMethod("transcript_quantification", "FLAMES.Pipeline", function(pipeline, reference_only) {
  if ((!missing(reference_only) && reference_only) || is.na(pipeline@novel_isoform_annotation)) {
    annotation <- pipeline@annotation
  } else {
    annotation <- pipeline@novel_isoform_annotation
  }
  pipeline_class <- switch(class(pipeline),
    "FLAMES.Pipeline" = "bulk",
    "FLAMES.SingleCellPipeline" = "sc_single_sample",
    "FLAMES.MultiSampleSCPipeline" = "sc_multi_sample"
  )
  # TODO: refactor quantify_transcript to take file paths from pipeline
  x <- quantify_transcript(
    annotation = annotation,
    outdir = pipeline@outdir,
    config = pipeline@config,
    pipeline = pipeline_class,
    samples = names(pipeline@fastq)
  )

  # Save experiment result to RDS file and store the file path
  saveRDS(x, pipeline@experiment)

  return(pipeline)
})

#' Index the reference genome for minimap2
#'
#' @description Calls minimap2 to index the reference genome.
#' @param pipeline A FLAMES.Pipeline object.
#' @param path The file path to save the minimap2 index. If not provided, it will be saved
#'   to the output directory with the name "genome.mmi".
#' @param additional_args (optional) Additional arguments to pass to minimap2.
#' @return A \code{SummarizedExperiment} object, a \code{SingleCellExperiment} object,
#' or a list of \code{SingleCellExperiment} objects.
#' @examples
#' pipeline <- example_pipeline(type = "BulkPipeline")
#' pipeline <- index_genome(pipeline)
#' @export
setGeneric("index_genome", function(pipeline, path, additional_args = c("-k", "14")) {
  standardGeneric("index_genome")
})
#' @rdname index_genome
#' @export
setMethod("index_genome", "FLAMES.Pipeline", function(pipeline, path, additional_args = c("-k", "14")) {
  if (missing(path) || !is.character(path) || is.na(path)) {
    path <- file.path(pipeline@outdir, "genome.mmi")
  }

  cmd <- paste(
    shQuote(pipeline@minimap2), "-d", shQuote(path), shQuote(pipeline@genome_fa),
    paste(shQuote(additional_args), collapse = " ")
  )
  minimap2_status <- system(cmd, intern = FALSE)
  check_status_code(minimap2_status, cmd, "Minimap2")

  pipeline@genome_mmi <- path
  return(pipeline)
})

# Deprecated functions
#' Pipeline for bulk long read RNA-seq data processing (deprecated)
#'
#' @description This function is deprecated. Use \code{\link{BulkPipeline}} instead.
#'
#' @param annotation The file path to the annotation file in GFF3 / GTF format.
#' @param fastq Path to the FASTQ file or a directory containing FASTQ files. Each file
#'   will be processed as an individual sample.
#' @param outdir Path to the output directory. If it does not exist, it will be created.
#' @param genome_fa The file path to the reference genome in FASTA format.
#' @param minimap2 (optional) The path to the minimap2 binary. If not provided, FLAMES will
#'   use a copy from bioconda via \code{basilisk}.
#' @param config_file Path to the JSON configuration file. See \code{\link{create_config}} for creating one.
#'
#' @return A \code{SummarizedExperiment} object containing the transcript counts.
#' @seealso
#' \code{\link{BulkPipeline}} for the new pipeline function.
#' \code{\link{SingleCellPipeline}} for single cell pipelines,
#' \code{\link{MultiSampleSCPipeline}} for multi sample single cell pipelines.
#'
#' @examples
#' outdir <- tempfile()
#' dir.create(outdir)
#' # simulate 3 samples via sampling
#' reads <- ShortRead::readFastq(
#'   system.file("extdata", "fastq", "musc_rps24.fastq.gz", package = "FLAMES")
#' )
#' dir.create(file.path(outdir, "fastq"))
#' ShortRead::writeFastq(reads[1:100],
#'   file.path(outdir, "fastq/sample1.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' reads <- reads[-(1:100)]
#' ShortRead::writeFastq(reads[1:100],
#'   file.path(outdir, "fastq/sample2.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' reads <- reads[-(1:100)]
#' ShortRead::writeFastq(reads,
#'   file.path(outdir, "fastq/sample3.fq.gz"),
#'   mode = "w", full = FALSE
#' )
#' # prepare the reference genome
#' genome_fa <- file.path(outdir, "rps24.fa")
#' R.utils::gunzip(
#'   filename = system.file("extdata", "rps24.fa.gz", package = "FLAMES"),
#'   destname = genome_fa, remove = FALSE
#' )
#' se <- bulk_long_pipeline(
#'   fastq = file.path(outdir, "fastq"),
#'   annotation = system.file("extdata", "rps24.gtf.gz", package = "FLAMES"),
#'   outdir = outdir, genome_fa = genome_fa,
#'   config_file = create_config(outdir, type = "sc_3end", threads = 1, no_flank = TRUE)
#' )
#' se
#' @export
bulk_long_pipeline <- function(
    annotation, fastq, outdir, genome_fa,
    minimap2 = NULL, config_file) {
  message("bulk_long_pipeline() is deprecated. Use BulkPipeline() instead.")
  pipeline <- BulkPipeline(
    config_file = config_file,
    outdir = outdir,
    fastq = fastq,
    annotation = annotation,
    genome_fa = genome_fa,
    minimap2 = minimap2
  )
  pipeline <- run_FLAMES(pipeline)
  saveRDS(pipeline, file.path(outdir, "pipeline.rds"))
  message("Pipeline saved to ", file.path(outdir, "pipeline.rds"))
  if (length(pipeline@last_error) == 0) {
    return(experiment(pipeline))
  } else {
    warning("Returning pipeline object instead of experiment due to errors.")
    message("You can resume the pipeline after resolving the errors with resume_FLAMES(pipeline)")
    return(pipeline)
  }
}

#' Plot pipeline step durations
#'
#' @description This function creates a horizontal bar plot showing the duration
#' of each pipeline step using ggplot2.
#' @param x A FLAMES.Pipeline object.
#' @return A ggplot2 object.
#' @examples
#' pipeline <- example_pipeline("BulkPipeline")
#' pipeline <- run_FLAMES(pipeline)
#' plot_durations(pipeline)
#' @importFrom ggplot2 ggplot aes geom_col coord_flip labs theme_minimal theme element_text scale_y_continuous geom_text
#' @export
setGeneric("plot_durations", function(x) standardGeneric("plot_durations"))
#' @rdname plot_durations
#' @export
setMethod("plot_durations", "FLAMES.Pipeline", function(x) {
  durations <- x@durations

  if (length(durations) == 0) {
    stop("No step durations available. Run the pipeline first.")
  }

  # Convert durations to numeric values in seconds
  duration_secs <- as.numeric(durations)

  # Determine appropriate time units for display
  max_duration <- max(duration_secs)
  if (max_duration < 60) {
    # Use seconds
    duration_values <- duration_secs
    time_unit <- "seconds"
  } else if (max_duration < 3600) {
    # Use minutes
    duration_values <- duration_secs / 60
    time_unit <- "minutes"
  } else if (max_duration < 86400) {
    # Use hours
    duration_values <- duration_secs / 3600
    time_unit <- "hours"
  } else {
    # Use days
    duration_values <- duration_secs / 86400
    time_unit <- "days"
  }

  # Create data frame for ggplot
  plot_data <- data.frame(
    step = factor(names(durations), levels = rev(names(durations))),
    duration = duration_values,
    stringsAsFactors = FALSE
  )

  # Create ggplot
  p <- ggplot2::ggplot(plot_data, ggplot2::aes(x = step, y = duration)) +
    ggplot2::geom_col(fill = "steelblue", width = 0.7) +
    ggplot2::coord_flip() +
    ggplot2::labs(
      title = "FLAMES Pipeline Step Durations",
      x = "Pipeline Step",
      y = paste("Duration (", time_unit, ")", sep = "")
    ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(
      plot.title = ggplot2::element_text(hjust = 0.5, size = 14, face = "bold"),
      axis.text = ggplot2::element_text(size = 11),
      axis.title = ggplot2::element_text(size = 12)
    ) +
    ggplot2::scale_y_continuous(expand = c(0, 0, 0.1, 0)) +
    ggplot2::geom_text(
      ggplot2::aes(label = sprintf("%.1f", duration)),
      hjust = 1.1,
      color = "white",
      fontface = "bold",
      size = 3.5
    )

  return(p)
})
