#' @name DeeDeeExperiment-methods
#'
#' @title Methods for [DeeDeeExperiment] objects
#'
#' @aliases
#' getDEAInfo
#' getDEAInfo<-
#' getDEANames
#' renameDEA
#' addDEA
#' removeDEA
#' getDEA
#' getDEAList
#' addScenarioInfo
#' getFEAInfo
#' getFEAInfo<-
#' getFEANames
#' renameFEA
#' addFEA
#' removeFEA
#' getFEA
#' getFEAList
#' linkDEAandFEA
#'
#' @description
#' The [DeeDeeExperiment()] class provides a family of methods to get
#' and set DE-related information and functional enrichment results in
#' [DeeDeeExperiment] objects.
#'
#' @param x A [DeeDeeExperiment()] object
#' @param value Replacement value for replacement methods.
#' @param dea A named list of DE results, in any of the formats supported by
#' the package (currently: results from DESeq2, edgeR, limma).
#' @param dea_name Character value, specifying the name of the DE analysis to
#' get or remove, or match against (e.g., to fetch associated FEA results), or
#' to which additional context and information can be attached
#' @param verbose Logical, whether or not to display warnings. If TRUE,
#' warnings/messages will be displayed. If FALSE, the function runs silently
#' @param extra_rd A character vector of additional columns from rowData(x)
#' to include. It defaults to c("gene_id", "SYMBOL").
#' @param type A character string referring to the type of object returned by
#' `getDEA()`. It defaults to `DFrame`, but can also take the value of `data.frame`
#' @param old_name A character vector of existing DEA names to be renamed in a
#' `DeeDeeExperiment` object
#' @param new_name A character vector with new names to assign to existing DEA
#' names in a `DeeDeeExperiment` object. It must be the same length of
#' `old_name`, and contains unique values that don't overlap with existing DEA
#' names.
#' @param fea A named list of Functional Enrichment results. Each element can be
#' either a data.frame (currently supports results from `topGO`, `enrichR`,
#' `gProfiler`, `fgsea`, `gsea`, `DAVID`, and output of `GeneTonic` shakers),
#' or an `enrichResult`/`gseaResult` objects (currently supports
#' `clusterProfiler`)
#' @param fea_name Character value, specifying the name of the functional
#' enrichment result to add or remove
#' @param de_name A character string to explicitly specify the name of the de
#' result this fea should be linked to. If not provided, the function will
#' attempt to match fea names to de results automatically.
#' @param fe_name A character string giving a name to the FE results.
#' @param remove_linked_fea A logical, specifying whether to remove or not the
#' linked FEA when a DEA results is removed
#' @param fea_tool A character string indicating the FEA tool used. It can take
#' any of the following values : "topGO", "clusterProfiler", "GeneTonic",
#' "DAVID", "gsea", "fgsea", "enrichr", "gProfiler". When not specified, it
#' defaults to "auto" and the tool is inferred automatically based on the input.
#' @param force A logical, indicating whether to overwrite results when
#' introducing the same results name. It defaults to FALSE.
#' @param format A character string, specifying the DEA/FEAs output format.
#' It takes either "minimal" to return only essential columns
#' (e.g. log2FC, p-value, adjusted p-value for DEAs,
#' or gs_id, gs_description, gs_pvalue, gs_genes... for FEAs), or "original" to
#' return the full result object. It defaults to "minimal"
#' @param info A character vector, containing contextual information about the
#' specified DE analysis. It defaults to NULL
#'
#' @return Return value varies depending on the individual methods, as described
#' below.
#'
#' @details
#'
#' DEAs
#'
#' * `getDEAInfo` and `getDEAInfo<-` are the methods to get and set the `dea`
#' information as a whole. These methods return `DeeDeeExperiment` objects.
#' * `getDEANames` returns the names of the available DE contrasts in
#' `DeeDeeExperiment` objects.
#' * `renameDEA` is the method to rename one or multiple DEAs stored in a
#' `DeeDeeExperiment` object.
#' * `addDEA` and `removeDEA` are used to respectively add or remove
#' DE-results items. These methods also return `DeeDeeExperiment` objects, with
#' updated content in the `dea` slot.
#' * `dea` and `getDEAList` retrieve the DEA information, as well as some
#' extra rowData information and provide this as a `DataFrame` object
#' (for a specific analysis) or as a list, with one element for each reported
#' analysis.
#' * `addScenarioInfo` is the method to add user defined contextual
#' information for a specific DE analysis.
#' It allows users to attach free-text notes to a specific DEA results that
#' stored in a `DeeDeeExperiment` object. This information can include any other
#' relevant information to help document that DEA scenario.
#' This context is stored in the `dea` slot under the name `scenario_info`,
#' which is not a default element in `dea`.
#'
#' FEAs
#'
#' * `getFEAInfo` and `getFEAInfo<-` are the methods to get and set the `fea`
#' information as a whole. These methods return `DeeDeeExperiment` objects.
#' * `getFEANames` returns the names of the available enrichment results in
#' `DeeDeeExperiment` objects.
#' * `renameFEA` is the method to rename one or multiple FEAs stored in a
#' `DeeDeeExperiment` object.
#' * `addFEA` and `removeFEA` are used to respectively add or remove
#' functional enrichment results items. These methods also return
#' `DeeDeeExperiment` objects, with updated content in the `fea` slot.
#' * `fea` is the method to retrieve FE results stored in a `DeeDeeExperiment`
#' object for a specific contrast, as a standardized format similar to the
#' output of `GeneTonic` shakers.
#' * `getFEAList` is the method that retrieves FEA results as a list. if the
#' `dea_name` is indicated, the method will return only FEAs linked to that
#' `dea_name`, otherwise it returns all FEAs in the `fea` slot.
#' * `linkDEAandFEA` is the method that allows the user to manually link a
#' FEA result to a specific DEA result.
#'
#' * `show` is the method to nicely print out the information of a
#' `DeeDeeExperiment` object.
#' * `summary` is the method to print a summary of the available DE and FE
#' results in a `DeeDeeExperiment` object.
#'
#' @examples
#' data("de_named_list", package = "DeeDeeExperiment")
#' data("topGO_results_list", package = "DeeDeeExperiment")
#' library("SummarizedExperiment")
#'
#' rd_macrophage <- DataFrame(
#'   gene_id = rownames(de_named_list$ifng_vs_naive)
#' )
#' rownames(rd_macrophage) <- rownames(de_named_list$ifng_vs_naive)
#' se_macrophage_noassays <- SummarizedExperiment(
#'   assays = SimpleList(),
#'   rowData = rd_macrophage
#' )
#'
#' # creating a `DeeDeeExperiment`
#' dde <- DeeDeeExperiment(
#'   se_macrophage_noassays,
#'   de_results = de_named_list
#' )
#' dde
#'
#' new_del <- list(
#'   ifng2 = de_named_list$ifng_vs_naive,
#'   ifngsalmo2 = de_named_list$ifngsalmo_vs_naive
#' )
#'
#' # add a new (set of) DE result(s)
#' dde_new <- addDEA(dde, new_del)
#' dde_new
#'
#' # removing DEAs
#' dde_removed <- removeDEA(dde, "ifng_vs_naive")
#' dde_removed
#'
#' # add a new (set of) FE result(s)
#' dde_new <- addFEA(dde, fea = topGO_results_list)
#'
#' # removing FEAs
#' dde_rem <- removeFEA(dde_new, "ifng_vs_naive")
#'
#' # display available DEAs
#' getDEANames(dde)
#'
#' # display available FEAs
#' getFEANames(dde)
#'
#' # print a summary of the available DEAs and FEAs
#' summary(dde, FDR = 0.01)
#'
#' # rename DEA
#' dde_new <- renameDEA(dde_new,
#'   old_name = "salmonella_vs_naive",
#'   new_name = "Salmo_vs_Naive_renamed"
#' )
#'
#' # assign DEA to FEA
#'
#' dde_new <- linkDEAandFEA(dde_new,
#'   dea_name = "ifngsalmo_vs_naive",
#'   fea_name = "ifngsalmo_vs_naive"
#' )
#'
NULL


# dea slot - get & set ---------------------------------------------------------

## getDEAInfo --------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getDEAInfo",
          signature = "DeeDeeExperiment",
          definition = function(x) {
            x@dea
          }
)

## getDEAInfo <- -----------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setReplaceMethod("getDEAInfo",
                 signature = c("DeeDeeExperiment", "ANY"),
                 definition = function(x, value) {
                   x@dea <- value
                   validObject(x)
                   x
                 }
)


# dea info - add, remove, get --------------------------------------------------

## getDEANames -------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getDEANames",
          signature = "DeeDeeExperiment",
          definition = function(x) {
            names(getDEAInfo(x))
          }
)

## renameDEA ------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("renameDEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                old_name,
                                new_name) {
            # check uniqueness of new names, and that they don't overlap with
            # existing ones

            if (!is.character(old_name) || length(old_name) == 0) {
              stop("'old_name' must be a non empty character vector!")
            }

            if (!is.character(new_name) || length(new_name) == 0) {
              stop("'new_name' must be a non empty character vector!")
            }

            deas <- getDEAInfo(x)
            current_names <- getDEANames(x)
            if (length(current_names) == 0) {
              stop("No DEA results found")
            }

            if (length(old_name) != length(new_name)) {
              stop("'old_name' and 'new_name' must be the same length!")
            }

            matching_index <- match(old_name, current_names)

            if (any(is.na(matching_index))) {
              missing_names <- old_name[is.na(matching_index)]
              stop(
                "The following DEA names where not found in dea slot:",
                paste(missing_names, collapse = ", ")
              )
            }

            if (anyDuplicated(new_name)) {
              stop("New names must be unique!")
            }

            overlapping_names <- intersect(new_name, current_names)
            if (length(overlapping_names) > 0) {
              stop(
                "New names overlap with existing DEA names: ",
                paste(overlapping_names, collapse = ", ")
              )
            }

            names(deas)[matching_index] <- new_name
            x@dea <- deas

            rd <- rowData(x)
            rd_colnames <- colnames(rd)
            suffix <- c("_log2FoldChange", "_pvalue", "_padj")

            old_col <- unlist(lapply(old_name,
                                     function(prefix) paste0(prefix, suffix)),
                              use.names = FALSE)
            new_col <- unlist(lapply(new_name,
                                     function(prefix) paste0(prefix, suffix)),
                              use.names = FALSE)

            for (i in seq_along(old_col)) {
              if (old_col[i] %in% rd_colnames) {
                colnames(rd)[which(rd_colnames == old_col[i])] <- new_col[i]
              }
            }

            rowData(x) <- rd

            # also rename in fea slot in there is a linked fea

            fea_names <- getFEANames(x)

            for (fea in fea_names) {
              current_link <- getFEAInfo(x)[[fea]][["de_name"]]
              if (!is.null(current_link) && current_link %in% old_name) {
                new_index <- match(current_link, old_name)
                updated_name <- new_name[new_index]

                getFEAInfo(x)[[fea]][["de_name"]] <- updated_name
              }
            }

            cli::cli_alert_success(
              "Renamed DEA entries: {.val {old_name}} to {.val {new_name}}"
            )

            validObject(x)
            return(x)
          }
)


## addDEA ---------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("addDEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea,
                                force = FALSE) {
            # dea must be named list
            if (is.null(names(dea))) {
              stop("All elements in dea list must have names!")
            }

            # check that names are all unique
            if (anyDuplicated(names(dea))) {
              stop("Names in dea must be unique!")
            }

            # unless force is TRUE

            new_names <- names(dea)
            existing_names <- names(getDEAInfo(x))

            overlapping_names <- intersect(new_names, existing_names)

            if (length(overlapping_names) > 0 && !force) {
              stop(
                "Names in 'dea' overlap with existing DEA results: ",
                paste(overlapping_names, collapse = ", "),
                ". Set force = TRUE to overwrite."
              )
            }


            # capture name inside the env where the func is called
            entry_name <- deparse(substitute(dea))

            # check and preocess dea
            dea <- .check_de_results(dea, entry_name)

            dea_contrasts <- getDEAInfo(x)

            # update rowData, naming them correctly
            for (i in names(dea)) {
              this_de <- dea[[i]]

              # do different things according to what these objects are
              if (is(this_de, "DESeqResults")) {
                # check for rowname mismatches
                rownames_x <- rownames(rowData(x))
                rownames_y <- rownames(this_de)
                mismatched_rows <- sum(!rownames_x %in% rownames_y)

                mismatch_percent <- (mismatched_rows / length(rownames_x)) * 100

                if (mismatch_percent > 50) {
                  warning(
                    "A Total number of ",
                    mismatched_rows,
                    " mismatched rows detected between `rownames(rowData(se))`",
                    " and rownames for the following dea element: ",
                    i,
                    "Unmatched genes will have NA values in rowData. ",
                    ". Consider synchronizing your rownames in both se and","
            de_results elements."
                  )
                }

                # we align de res with se
                matched_ids <- match(rownames(x), rownames(this_de))
                # only valid indices
                valid_matches <- !is.na(matched_ids)

                # Pre-fill rowData with NA
                # assign values only for matched indices, to have on both sides
                # the same length. we keep NA for unmatched genes
                x <- .fill_rowdata_with_dea(sce = x,
                                              de_name = i,
                                              de_res = this_de,
                                              de_cols = c(logFC = "log2FoldChange",
                                                          pval = "pvalue",
                                                          padj = "padj"),
                                              valid_matches = valid_matches,
                                              matched_ids = matched_ids)


                dea_contrasts[[i]] <- list(
                  alpha = metadata(this_de)$alpha,
                  lfcThreshold = metadata(this_de)$lfcThreshold,
                  metainfo_logFC = mcols(this_de)$description[colnames(this_de) == "log2FoldChange"],
                  metainfo_pvalue = mcols(this_de)$description[colnames(this_de) == "pvalue"],
                  original_object = this_de,
                  package = "DESeq2",
                  package_version = packageVersion("DESeq2")
                )
              } else if (is(this_de, "DGEExact") || is(this_de, "DGELRT")) {
                # check for rowname mismatches
                rownames_x <- rownames(rowData(x))
                rownames_y <- rownames(this_de)
                mismatched_rows <- sum(!rownames_x %in% rownames_y)

                mismatch_percent <- (mismatched_rows / length(rownames_x)) * 100

                if (mismatch_percent > 50) {
                  warning(
                    "A Total number of ",
                    mismatched_rows,
                    " mismatched rows detected between `rownames(rowData(se))`",
                    " and rownames for the following dea element: ",
                    i,
                    "Unmatched genes will have NA values in rowData. ",
                    ". Consider synchronizing your rownames in both se and",
                    " de_results elements."
                  )
                }

                res_tbl <- topTags(
                  this_de,
                  n = nrow(this_de),
                  sort.by = "none"
                )$table

                # p value different from NA respect the 0-1 interval
                stopifnot(all(na.omit(res_tbl$PValue <= 1)) &
                            all(na.omit(res_tbl$PValue > 0)))

                # identify the logFC cols
                logFC_cols <- grep("^logFC", colnames(res_tbl), value = TRUE)

                # we align de res with se
                matched_ids <- match(rownames(x), rownames(res_tbl))
                # only valid indices
                valid_matches <- !is.na(matched_ids)

                # pre-fill rowData with NA the assign the corresponding values
                # only for matched indices for logFC, accounting for the fact
                # that the logFC column name in edgeR
                # depends on whether we have 1 or multiple contrasts
                for (j in logFC_cols) {
                  suffix <- ifelse(j == "logFC", "", paste0("_",
                                                            sub("^logFC[.]*",
                                                            "", j)))
                  this_de_name <- paste0(i, suffix)

                  x <- .fill_rowdata_with_dea(sce = x,
                                                de_name = this_de_name,
                                                de_res = res_tbl,
                                                de_cols = c(logFC = j,
                                                            pval = "PValue",
                                                            padj = "FDR"),
                                                valid_matches = valid_matches,
                                                matched_ids = matched_ids)
                }

                # store metadata
                dea_contrasts[[i]] <- list(
                  alpha = NA,
                  lfcThreshold = NA,
                  metainfo_logFC = res_tbl$comparison,
                  metainfo_pvalue = NA,
                  original_object = this_de,
                  package = "edgeR",
                  package_version = packageVersion("edgeR")
                )
              } else if (is(this_de, "MArrayLM")) {
                # check for rowname mismatches
                rownames_x <- rownames(rowData(x))
                rownames_y <- rownames(this_de)
                mismatched_rows <- sum(!rownames_x %in% rownames_y)

                mismatch_percent <- (mismatched_rows / length(rownames_x)) * 100

                if (mismatch_percent > 50) {
                  warning(
                    "A Total number of ",
                    mismatched_rows,
                    " mismatched rows detected between `rownames(rowData(se))`",
                    " and rownames for the following dea element: ",
                    i,
                    "Unmatched genes will have NA values in rowData. ",
                    ". Consider synchronizing your rownames in both se and",
                    " de_results elements."
                  )
                }

                res_tbl <- topTable(
                  this_de,
                  coef    = 2,
                  number  = nrow(this_de),
                  sort.by = "none"
                )

                # p value different from NA respect the 0-1 interval
                stopifnot(all(na.omit(res_tbl$P.Value <= 1)) &
                            all(na.omit(res_tbl$P.Value > 0)))

                # we align de res with se
                matched_ids <- match(rownames(x), rownames(res_tbl))
                # only valid indices
                valid_matches <- !is.na(matched_ids)

                x <- .fill_rowdata_with_dea(sce = x,
                                              de_name = i,
                                              de_res = res_tbl,
                                              de_cols = c(logFC = "logFC",
                                                          pval = "P.Value",
                                                          padj = "adj.P.Val"),
                                              valid_matches = valid_matches,
                                              matched_ids = matched_ids)

                # store metadata
                dea_contrasts[[i]] <- list(
                  alpha = NA,
                  lfcThreshold = NA,
                  metainfo_logFC = NA,
                  metainfo_pvalue = NA,
                  original_object = this_de,
                  package = "limma",
                  package_version = packageVersion("limma")
                )
              } else if (is(this_de, "data.frame")) {
                # contain the right columns
                stopifnot(all(c("log2FoldChange", "pvalue", "padj") %in%
                                colnames(this_de)))
                # check for rowname mismatches
                rownames_x <- rownames(rowData(x))
                rownames_y <- rownames(this_de)
                mismatched_rows <- sum(!rownames_x %in% rownames_y)

                mismatch_percent <- (mismatched_rows / length(rownames_x)) * 100

                if (mismatch_percent > 50) {
                  warning(
                    "A Total number of ",
                    mismatched_rows,
                    " mismatched rows detected between `rownames(rowData(se))`",
                    " and rownames for the following dea element: ",
                    i,
                    "Unmatched genes will have NA values in rowData. ",
                    ". Consider synchronizing your rownames in both se and",
                    " de_results elements."
                  )
                }

                # p value different from NA respect the 0-1 interval
                stopifnot(all(na.omit(this_de$pvalue <= 1)) &
                            all(na.omit(this_de$pvalue > 0)))

                # we align de res with se
                matched_ids <- match(rownames(x), rownames(this_de))
                # only valid indices
                valid_matches <- !is.na(matched_ids)

                x <- .fill_rowdata_with_dea(sce = x,
                                            de_name = i,
                                            de_res = this_de,
                                            de_cols = c(logFC = "log2FoldChange",
                                                        pval = "pvalue",
                                                        padj = "padj"),
                                            valid_matches = valid_matches,
                                            matched_ids = matched_ids)

                # store metadata
                dea_contrasts[[i]] <- list(
                  alpha = NA,
                  lfcThreshold = NA,
                  metainfo_logFC = NA,
                  metainfo_pvalue = NA,
                  original_object = this_de,
                  package = NA,
                  package_version = NA
                )
              } else {
                stop(
                  "The dea result class '", i,
                  "' is not recognized (supported classes: DESeqResults, ",
                  "MArrayLM, DGEExact and DGELRT), or data.frame ",
                  "with at least a 'log2FoldChange', 'pvalue' and 'padj' columns"
                )
              }
            }

            # update the dea slot
            getDEAInfo(x) <- dea_contrasts

            # check here the validity
            validObject(x)

            # return the object
            return(x)
          }
)


## removeDEA ------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("removeDEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea_name,
                                remove_linked_fea = FALSE) {
            # dea_name must be character
            if (!is.character(dea_name) || length(dea_name) == 0) {
              stop("'dea_name' must be a non empty character vector!")
            }

            # dea must be char vector
            deas <- names(getDEAInfo(x))

            deas_to_remove <- intersect(dea_name, deas)

            # warning() if nothing to remove
            if (length(deas_to_remove) == 0) {
              warning(
                "Some elements in 'dea_name' were not found among DEA results.\n",
                "Available results: ", paste(deas, collapse = ",")
              )
            }

            all_cols_to_remove <- unlist(lapply(deas_to_remove, function(i)
              paste0(i, c("_log2FoldChange", "_pvalue",
                          "_padj"))), use.names = FALSE)

            rowData(x) <-
              rowData(x)[, !(colnames(rowData(x)) %in% all_cols_to_remove)]
            # update the de slot
            getDEAInfo(x)[deas_to_remove] <- NULL

            feas <- getFEAInfo(x)

            ## fea is not removed by default, unless = TRUE
            if (remove_linked_fea) {
              removed_fea <- names(feas)[sapply(feas, function(fea)
                !is.null(fea[["de_name"]]) &&
                  fea[["de_name"]] %in% deas_to_remove)]
              # remove
              if (length(removed_fea) > 0) {
                feas[removed_fea] <- NULL
                getFEAInfo(x) <- feas
                cli::cli_alert_success(
                  "The following linked FEA entries were removed: {.val {paste(removed_fea, collapse = ', ')}} ")
              }
            }

            # unlink
            feas <- lapply(feas, function(fea) {
              if (!is.null(fea[["de_name"]]) &&
                  fea[["de_name"]] %in% deas_to_remove) {
                fea[["de_name"]] <- NULL
              }
              fea
            })
            getFEAInfo(x) <- feas

            # here check some validity?
            validObject(x)

            # return the object
            return(x)
          }
)


## getDEA -------------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getDEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea_name = NULL,
                                format = c("minimal", "original"),
                                extra_rd = NULL,
                                type = c("DFrame", "data.frame"),
                                verbose = FALSE) {
            # checks
            if (!is.null(extra_rd) && !is.character(extra_rd)) {
              stop("'extra_rd' must be a character vector!")
            }

            deas <- getDEAInfo(x)
            dea_names <- names(deas)

            format <- match.arg(format)

            # if (!(format %in% c("minimal", "original"))) {
            #   stop(
            #     "'format' not supported. Please use 'minimal' to return the ",
            #     "essential columns, or 'original' to return the original object"
            #   )
            # }

            if (is.null(dea_name)) {
              if (length(dea_names) == 0) {
                stop("No DEA results found")
              }

              warning(
                "'dea_name' was not specified. Returning the 1st entry: ",
                dea_names[1]
              )

              dea_name <- dea_names[1]
            }


            if (!is.character(dea_name) || length(dea_name) != 1) {
              stop("'dea_name' must be a single character string!")
            }

            if (!(dea_name %in% dea_names)) {
              stop(
                "Could not find '", dea_name, "' among DEA results.\n",
                "Available results: ", paste(dea_names, collapse = ",")
              )
            }

            type <- match.arg(type)

            if (!is.character(type) || length(type) != 1) {
              "'type' must be a single character string!"
            }

            # if (!type %in% c("DFrame", "data.frame")) {
            #   stop("'type' must be 'DFrame' or 'data.frame'!")
            # }


            if (format == "minimal") {
              rd_info <- paste0(
                dea_name,
                c("_log2FoldChange", "_pvalue", "_padj")
              )

              extra_info <- extra_rd
              # drop if missing
              extra_cols <- extra_info[extra_info %in% colnames(rowData(x))]
              all_cols <- c(extra_cols, rd_info)

              overlap <- intersect(extra_info, rd_info)

              if (length(overlap) > 0) {
                stop(
                  "The following `extra_rd` are already part of the core `dea`",
                  " columns and should not be repeated: ",
                  paste(overlap, collapse = ", ")
                )
              }

              if (verbose && length(setdiff(extra_info, extra_cols)) > 0) {
                warning(
                  "Some 'extra_rd' are not available in rowData: ",
                  paste(setdiff(extra_info, extra_cols), collapse = ", ")
                )
              }

              # check for missing columns, for a more precise feedback on
              # the error
              missing_cols <- rd_info[!rd_info %in% colnames(rowData(x))]

              # maybe check for rowname mismatches potential gene version issue?
              rownames_x <- rownames(rowData(x))
              rownames_y <-
                rownames(getDEAInfo(x)[[dea_name]][["original_object"]])
              mismatched_rows <- sum(!rownames_x %in% rownames_y)

              affected_deas <- character()
              if (mismatched_rows > 0) {
                affected_deas <- c(affected_deas, dea_name)
              }

              if (length(affected_deas) > 0) {
                if (verbose) {
                  warning(
                    "Mismatch detected between `rownames(rowData(x))` and ",
                    "rownames for the following dea element(s): ",
                    paste(unique(affected_deas), collapse = ", ")
                  )
                }
              }

              if (length(missing_cols) > 0) {
                stop(
                  "The following columns are missing: ",
                  paste(missing_cols, collapse = ", ")
                )
              }

              out <- rowData(x)[, all_cols]
              if (type == "data.frame") {
                out <- as.data.frame(out)
              }
            } else if (format == "original") {
              out <- getDEAInfo(x)[[dea_name]][["original_object"]]
            }
            return(out)
          }
)



## getDEAList ----------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getDEAList",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                format = c("minimal", "original"),
                                verbose = FALSE) {
            format <- match.arg(format)
            # if (!(format %in% c("minimal", "original"))) {
            #   stop(
            #     "'format' not supported. Please use 'minimal' to return the ",
            #     "essential columns, or 'original' to return the original object"
            #   )
            # }

            deas <- getDEAInfo(x)
            dea_names <- names(deas)

            dea_list <- list()
            affected_deas <- character()

            for (i in dea_names) {
              dea_list[[i]] <- as.data.frame(
                getDEA(x, dea_name = i, format = format, verbose = verbose)
              )

              if (format == "minimal") {
                colnames(dea_list[[i]]) <- c("log2FoldChange", "pvalue", "padj")
              }
            }

            # no need to check mismatch since the warnings will be triggered
            # from dea

            return(dea_list)
          }
)


## addScenarioInfo -----------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("addScenarioInfo",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea_name,
                                info = NULL,
                                force = FALSE) {
            # checks on dea_name
            if (!is.character(dea_name) || length(dea_name) != 1) {
              stop("'dea_name' must be a single character string!")
            }

            dea_names <- getDEANames(x)
            existing_info <- getDEAInfo(x)[[dea_name]][["scenario_info"]]

            # checks on info
            if (!is.null(info) && !is.character(info)) {
              stop(
                "'info' must be a character vector (e.g. one or more strings)"
              )
            }

            if (!(dea_name %in% dea_names)) {
              stop(
                "'dea_name'", dea_name, "not found among DEA results.\n",
                "Available results: ", paste(dea_names, collapse = ",")
              )
            }

            if (!is.null(existing_info) && !force) {
              stop(
                "Existing scenario_info for '", dea_name, "' already exists.",
                "Set force = TRUE to overwrite"
              )
            }

            # when both info and existing_info are null -> do nothing

            getDEAInfo(x)[[dea_name]][["scenario_info"]] <- info

            # update object
            validObject(x)

            return(x)
          }
)




# fea slot - get & set ---------------------------------------------------------

## getFEAInfo --------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getFEAInfo",
          signature = "DeeDeeExperiment",
          definition = function(x) {
            x@fea
          }
)

## getFEAInfo <- -----------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setReplaceMethod("getFEAInfo",
                 signature = c("DeeDeeExperiment", "ANY"),
                 definition = function(x, value) {
                   x@fea <- value
                   validObject(x)
                   x
                 }
)


# fea info - add, remove, get --------------------------------------------------

## getFEANames -------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getFEANames",
          signature = "DeeDeeExperiment",
          definition = function(x) {
            names(getFEAInfo(x))
          }
)

## renameFEA ------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("renameFEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                old_name,
                                new_name) {
            # check uniqueness of new names, and that they don't overlap with
            # existing ones

            if (!is.character(old_name) || length(old_name) == 0) {
              stop("'old_name' must be a non empty character vector!")
            }

            if (!is.character(new_name) || length(new_name) == 0) {
              stop("'new_name' must be a non empty character vector!")
            }

            feas <- getFEAInfo(x)
            current_names <- getFEANames(x)
            if (length(current_names) == 0) {
              stop("No FEA results found")
            }

            if (length(old_name) != length(new_name)) {
              stop("'old_name' and 'new_name' must be the same length!")
            }

            matching_index <- match(old_name, current_names)

            if (any(is.na(matching_index))) {
              missing_names <- old_name[is.na(matching_index)]
              stop(
                "The following FEA names where not found in fea slot:",
                paste(missing_names, collapse = ", ")
              )
            }

            if (anyDuplicated(new_name)) {
              stop("New names must be unique!")
            }

            overlapping_names <- intersect(new_name, current_names)
            if (length(overlapping_names) > 0) {
              stop(
                "New names overlap with existing FEA names: ",
                paste(overlapping_names, collapse = ", ")
              )
            }

            names(feas)[matching_index] <- new_name
            x@fea <- feas

            cli::cli_alert_success(
              "Renamed FEA entries: {.val {old_name}} to {.val {new_name}}"
            )

            validObject(x)

            return(x)
          }
)

## addFEA ---------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod(
  "addFEA",
  signature = c("DeeDeeExperiment"),
  definition = function(x,
                        fea,
                        de_name = NA_character_,
                        fe_name = NULL,
                        fea_tool = "auto",
                        force = FALSE,
                        verbose = FALSE) {

    if (!is.character(de_name) || length(de_name) != 1) {
      stop("'de_name' must be a single character string or NA_character_")
    } # should it be a vector of different de_name???


    # allowed fea_tools

    allowed_fea_tools <- c(
      "auto", "topGO", "clusterProfiler", "GeneTonic",
      "DAVID", "gsea", "fgsea", "enrichr", "gProfiler"
    )

    if (!is.character(fea_tool)) {
      stop("fea_tool should be a character vector!")
    }

    if (!all(fea_tool %in% allowed_fea_tools)) {
      stop("fea_tool should be one of the following: topGO, clusterProfiler,
      GeneTonic, DAVID, gsea, fgsea, enrichr, gProfiler")
    }

    # capture name inside the env where the func is called
    entry_name <- deparse(substitute(fea))

    # check and preocess fea
    fea_list <- .check_enrich_results(fea, entry_name)

    # check that names are all unique
    if (anyDuplicated(names(fea))) {
      stop("Names in dea must be unique!")
    }

    # check that names are all unique, and do not overlap with the existing ones
    # unless force is TRUE

    new_names <- names(fea_list)
    existing_names <- names(getFEAInfo(x))

    overlapping_names <- intersect(new_names, existing_names)

    if (length(overlapping_names) > 0 && !force) {
      stop(
        "Names in 'fea' overlap with existing FEA results: ",
        paste(overlapping_names, collapse = ", "),
        ". Set force = TRUE to overwrite."
      )
    }

    # get existing results in the fea slot
    fea_contrasts <- getFEAInfo(x)

    for (fe in names(fea_list)) {
      res_enrich <- fea_list[[fe]]
      if (!is.null(getDEAInfo(x)) && length(getDEAInfo(x)) > 0) {
        if (!is.na(de_name)) {
          if (de_name %in% names(getDEAInfo(x))) {
            de_res_name <- de_name
          } else {
            warning(
              "Provided 'de_name' ('",
              de_name,
              "') not found among DE results. Coercing into NA_character_")
            de_res_name <- NA_character_
          }
        } else {
          matched_name <- .match_fe_to_de(fe, names(getDEAInfo(x)))
          if (!is.na(matched_name) && matched_name %in% names(getDEAInfo(x))) {
            de_res_name <- matched_name
            if (fe != matched_name) {
              if (verbose) {
                cli::cli_alert_info(
                  "FEA {.val {fe}} matched to DE contrast {.val {matched_name}}"
                )
              }
            } else {
              # in case of the same name
              if (verbose) {
                cli::cli_alert_info(
                  "FEA {.val {fe}} matched directly to DE contrast {.val {matched_name}}"
                )
              }
            }
          } else {
            de_res_name <- NA_character_
            warning(
              "Could not match FEA '", fe, "' to any DE contrast.\n",
              "Available DE results: ", paste(names(getDEAInfo(x)),
                                              collapse = ", "), "\n",
              "Consider naming your enrich_results starting with one of the",
              " following prefixes:",
              " 'topGO_', 'clusterProfiler_','GeneTonic_', 'DAVID_','gsea_',",
              " 'fgsea_', 'enrichr_', 'gPro_', followed by the contrast name"
            )
          }
        }
      } else {
        de_res_name <- NA_character_
        warning(
          "Could not match FEA '",
          fe,
          "' to a DE contrast because no DE results were provided.\n")
      }


      n_fea <- length(fea_list)

      if (length(fea_tool) == 1 && fea_tool == "auto") {
        fea_tool_vec <- rep("auto", n_fea)
      } else if (length(fea_tool) == 1 && fea_tool %in% allowed_fea_tools) {
        fea_tool_vec <- rep(fea_tool, n_fea)
      } else if (length(fea_tool) == n_fea &&
                 all(fea_tool %in% allowed_fea_tools)) {
        fea_tool_vec <- fea_tool
      } else {
        stop(
          "'fea_tool' must be either: A single valid tool name (e.g.",
          paste(allowed_fea_tools, collapse = ", "),
          "or a character vector of length ",
          n_fea, " with tool names matching the FEA elements in order"
        )
      }

      names(fea_tool_vec) <- names(fea_list)

      this_tool <- fea_tool_vec[[fe]]

      if (this_tool == "auto") {
        # auto detect
        fe_tool <- .detect_fea_tool(res_enrich)
      } else {
        fe_tool <- this_tool
      }
      res_enrich_shaken <- NULL # default

      res_enrich_shaken <- .shake_enrich_res(res_enrich, fe_tool)

      if (is.null(res_enrich_shaken)) {
        cli::cli_alert_info(
          "No shaking method available for this functional enrichment results.
                            Returning only the original object.")
      }

      fea_contrast <- list(
        de_name = de_res_name,
        # links to de result
        fe_name = if (!is.null(fe_name)) fe_name else fe,
        shaken_results = res_enrich_shaken,
        # return shaken results for later use in GeneTonic
        original_object = res_enrich,
        fe_tool = fe_tool,
        fe_tool_version = if (
          fe_tool %in% loadedNamespaces()) packageVersion(fe_tool) else NA
      )

      fea_contrasts[[fe]] <- fea_contrast
    }
    # update the fea slot
    getFEAInfo(x) <- fea_contrasts
    # check here the validity
    validObject(x)
    # return the object
    return(x)
  }
)


## removeFEA ------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("removeFEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x, fea_name) {

            if (!is.character(fea_name) || length(fea_name) == 0) {
              stop("'fea_name' must be a non empty character vector!")
            }

            feas <- getFEANames(x)

            if (!all(fea_name %in% feas)) {
              stop(
                "Some elements in 'fea_name' were not found among FEA results.\n",
                "Available results: ", paste(feas, collapse = ",")
              )
            }

            feas_to_remove <- intersect(fea_name, feas)

            # warning() if nothing to remove
            if (length(feas_to_remove) == 0) {
              warning("No matching fea entries found to remove.")
            }

            for (i in feas_to_remove) {
              # update the fea slot
              getFEAInfo(x)[[i]] <- NULL
            }

            # here check some validity?
            validObject(x)

            # return the object
            return(x)
          }
)


## getFEA-------------------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getFEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                fea_name = NULL,
                                format = c("minimal", "original"),
                                verbose = FALSE) {

            format <- match.arg(format)

            fea_names <- getFEANames(x)

            if (is.null(fea_name)) {
              if (length(fea_names) == 0) {
                stop("No FEA results found")
              }

              warning(
                "'fea_name' was not specified. Returning the 1st entry: ",
                fea_names[1]
              )

              fea_name <- fea_names[1]
            }

            if (!is.character(fea_name) || length(fea_name) != 1) {
              stop("'fea_name' must be a single character string!")
            }

            if (!(fea_name %in% fea_names)) {
              stop(
                "Could not find '", fea_name, "' among FEA results.\n",
                "Available results: ", paste(fea_names, collapse = ",")
              )
            }

            if (format == "minimal") {
              fea <- getFEAInfo(x)[[fea_name]][["shaken_results"]]

              if (is.null(fea)) {
                if (verbose) {
                  warning(
                    "No shaken results available for '", fea_name,
                    "'. Returning original enrichment results instead."
                  )
                }
                fea <- getFEAInfo(x)[[fea_name]]$original_object
              }
            } else if (format == "original") {
              fea <- getFEAInfo(x)[[fea_name]][["original_object"]]
            }

            return(fea)
          }
)


## getFEAList ----------------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("getFEAList",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea_name = NULL,
                                format = c("minimal", "original"),
                                verbose = FALSE) {

            format <- match.arg(format)

            all_fea_names <- getFEANames(x)

            if (length(all_fea_names) == 0) {
              stop("No FEA results found")
            }

            matched_feas <- list()

            if (!is.null(dea_name) && (!is.character(dea_name) ||
                                       length(dea_name) != 1)) {
              stop("'dea_name' must be a single character string")
            }

            for (i in all_fea_names) {
              # catch the corresponding dea
              de_name <- getFEAInfo(x)[[i]][["de_name"]]

              # if dea_name is not indicated, return all feas
              # otherwise return only the specific feas associated with
              # that dea_name

              if (is.null(dea_name) ||
                  (!is.na(de_name) && de_name == dea_name)) {
                if (format == "minimal") {
                  fe_res <- getFEAInfo(x)[[i]][["shaken_results"]]

                  if (!is.null(fe_res)) {
                    matched_feas[[i]] <- fe_res
                  } else {
                    if (verbose) {
                      warning(
                        "No shaken results available for '",
                        i,
                        "'. Returning original enrichment results instead."
                      )
                    }

                    matched_feas[[i]] <- getFEAInfo(x)[[i]][["original_object"]]
                  }
                } else if (format == "original") {
                  matched_feas[[i]] <- getFEAInfo(x)[[i]][["original_object"]]
                }
              }
            }

            if (length(matched_feas) == 0) {
              if (!is.null(dea_name)) {
                warning("No FEA results found for '", dea_name, "'")
              } else {
                warning("No FEA results returned")
              }
            }

            return(matched_feas)
          }
)


## linkDEAandFEA -----------------------------------------------------------

#' @rdname DeeDeeExperiment-methods
#' @export
setMethod("linkDEAandFEA",
          signature = c("DeeDeeExperiment"),
          definition = function(x,
                                dea_name,
                                fea_name,
                                force = FALSE) {
            # check fea_name & dea_name are character
            if (!is.character(dea_name) || length(dea_name) == 0) {
              stop("'dea_name' must be a single character string!")
            }

            if (!is.character(fea_name) || length(fea_name) == 0) {
              stop("'fea_name' must be a non empty character vector!")
            }

            dea_names <- getDEANames(x)
            fea_names <- getFEANames(x)


            if (!(dea_name %in% dea_names)) {
              stop("DEA result: '", dea_name, "' not found")
            }

            for (fea in fea_name) {
              if (!(fea %in% fea_names)) {
                stop("FEA result: '", fea, "' not found")
              }

              # do we have existing link?
              current_de_name <- getFEAInfo(x)[[fea]][["de_name"]]

              if (!is.null(current_de_name) && !is.na(current_de_name) &&
                  current_de_name != dea_name) {
                if (!force) {
                  stop(
                    "FEA '", fea, "' is already linked to DEA '",
                    current_de_name,
                    "'. Use `force = TRUE` to overwrite"
                  )
                } else {
                  warning(
                    "FEA '", fea, "' was linked to DEA '", current_de_name,
                    ". Now linked to '", dea_name, "'"
                  )
                }
              }

              # assign
              cli::cli_alert_success(
                "Assigning DEA: {.val {dea_name}} to FEA {.val {fea}}"
              )

              getFEAInfo(x)[[fea]][["de_name"]] <- dea_name
            }

            validObject(x)
            return(x)
          }
)


# misc - show & more ------------------------------------------------------

#' @name DeeDeeExperiment-misc
#'
#' @title Miscellaneous DeeDeeExperiment methods
#'
#' @description
#' Miscellaneous methods for the \code{\link{DeeDeeExperiment}} class and its
#' descendants that do not fit into any other documentation category such as,
#' for example, show and summary methods.
#'
#' @param object a \code{\link{DeeDeeExperiment}} object
#' @param ... additional argument passed to the summary method.
#' Currently supports:
#' \itemize{
#'   \item `FDR`: Numeric, sets the significance threshold for subsetting
#'   differentially expressed genes based on adjusted p-values. Defaults to 0.05
#'   \item `show_scenario_info`: Logical; if \code{TRUE}, displays the
#'   associated scenario info for each DE contrast, if available.
#'   }
#' @return Returns NULL
NULL


## show ------------------------------------------------------------------------

#' @rdname DeeDeeExperiment-misc
#' @export
setMethod("show",
          signature = signature(object = "DeeDeeExperiment"),
          definition = function(object) {
            callNextMethod()
            cat(
              "dea(", length(object@dea), "): ",
              paste(names(object@dea), collapse = ", "), " \n",
              "fea(", length(object@fea), "): ",
              paste(names(object@fea), collapse = ", "),
              sep = ""
            )
            cat("\n")
          }
)


## summary ---------------------------------------------------------------------
#' @exportMethod summary
summary.DeeDeeExperiment <- function(object,
                                     FDR = 0.05,
                                     show_scenario_info = FALSE, ...) {
  # dea summary
  dea <- getDEAInfo(object)

  if (length(dea) > 0) {
    cat("DE Results Summary:\n")
    de_table <- data.frame(
      DEA_name = names(dea),
      Up = vapply(names(dea), function(contrast) {
        lfc_col <- paste0(contrast, "_log2FoldChange")
        padj_col <- paste0(contrast, "_padj")
        if (all(c(lfc_col, padj_col) %in% colnames(rowData(object)))) {
          lfc <- rowData(object)[[lfc_col]]
          padj <- rowData(object)[[padj_col]]
          sum(lfc > 0 & padj < FDR, na.rm = TRUE)
        } else {
          NA_integer_
        }
      }, integer(1)),
      Down = vapply(names(dea), function(contrast) {
        lfc_col <- paste0(contrast, "_log2FoldChange")
        padj_col <- paste0(contrast, "_padj")
        if (all(c(lfc_col, padj_col) %in% colnames(rowData(object)))) {
          lfc <- rowData(object)[[lfc_col]]
          padj <- rowData(object)[[padj_col]]
          sum(lfc < 0 & padj < FDR, na.rm = TRUE)
        } else {
          NA_integer_
        }
      }, integer(1)),
      FDR = rep(FDR, length(dea))
    )
    print(de_table, row.names = FALSE)

    cat("\n")
  } else {
    cat("No DEA results stored.\n\n")
  }
  # fea summary

  fea <- getFEAInfo(object)
  if (length(fea) > 0) {
    cat("FE Results Summary:\n")
    fea_table <- data.frame(
      FEA_Name = names(fea),
      Linked_DE = vapply(fea, function(object) {
        if (!is.null(object$de_name) && !is.na(object$de_name)) {
          object$de_name
        } else {
          "."
        }
      }, character(1)),
      FE_Type = vapply(fea, function(object) {
        if (!is.null(object$fe_tool)) {
          object$fe_tool
        } else {
          "Not Specified"
        }
      }, character(1)),
      Term_Number = vapply(fea, function(object) {
        if (!is.null(object$original_object)) {
          NROW(object$original_object)
        } else {
          NA_integer_
        }
      }, integer(1))
    )
    print(fea_table, row.names = FALSE)
  } else {
    cat("No FEA results stored.\n")
  }

  # scenario info (only if show_scenario_info is TRUE)
  if (show_scenario_info && length(dea) > 0) {
    cat("\nScenario Info:\n")
    missing <- character()
    for (de_name in names(dea)) {
      scenario_info <- dea[[de_name]][["scenario_info"]]
      if (!is.null(scenario_info)) {
        cat(" -", de_name, ":\n")

        wrapped_txt <- .basic_str_wrap(scenario_info,
                                       width = 80,
                                       indent = 1,
                                       exdent = 2
        )

        cat(paste(wrapped_txt, "\n"), "\n")
      } else {
        missing <- c(missing, de_name)
      }
    }

    if (length(missing) > 0) {
      cat("\nNo scenario info for:", paste(missing, collapse = ", "), "\n")
    }

    cat("\n")
  }
}

#' @rdname DeeDeeExperiment-misc
#'
#' @method summary DeeDeeExperiment
#'
#' @param FDR Numeric, sets the significance threshold for subsetting
#' differentially expressed genes based on adjusted p-values. Defaults to 0.05
#' @param show_scenario_info Logical; if TRUE, displays the associated scenario
#' info for each DE contrast, if available. Defaults to FALSE
#'
#' @export
setMethod("summary",
          signature = signature(object = "DeeDeeExperiment"),
          definition = summary.DeeDeeExperiment
)
