% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ONTdata.R
\name{readONTbam}
\alias{readONTbam}
\title{Load ONT BAM, decode MM/ML, and count modified vs. unmodified reads}
\usage{
readONTbam(
  bamfile,
  ref_gr = NULL,
  modif = "C+m?",
  prob_thresh = 0.5,
  genome = BSgenome.Hsapiens.UCSC.hg38,
  context = "CG",
  chr = NULL,
  region = NULL,
  synonymous = FALSE,
  parallel = FALSE,
  BPPARAM = NULL
)
}
\arguments{
\item{bamfile}{Path to an indexed ONT BAM with MM/ML tags.}

\item{ref_gr}{A \code{GRanges} of genomic cytosine positions to annotate.
If \code{NULL}, will be created via \code{selectCytosine()}
using \code{context}, \code{chr}, \code{region}.}

\item{modif}{Character vector of MM codes to treat as “modified”
(e.g. \code{"C+m?"}, \code{"C+h?"}, \code{"C+m."}).}

\item{prob_thresh}{Numeric in \[0,1\] — minimum ML probability to call a read
“modified.”}

\item{genome}{A \code{BSgenome} object such as \code{BSgenome.Hsapiens.UCSC.hg38}.
This is used to extract sequence context and must be loaded in advance.
\strong{Note:} When running on an HPC system,
please ensure that the required
\code{BSgenome} package is already installed and loaded in advance.}

\item{context}{Sequence context for \code{selectCytosine()}
(e.g. \code{"CG"}, \code{"CHG"}, \code{"CHH"}).}

\item{chr}{Chromosome names to restrict \code{selectCytosine()}.}

\item{region}{A \code{GRanges} to further subset \code{selectCytosine()}.}

\item{synonymous}{Logical (default: FALSE). If TRUE, include modified calls that match the specified context sequence (e.g. CGG),
even if the site was previously excluded due to deletion or mismatch.
For example, if a deletion occurs at position 234523, but the surrounding context still forms CGG,
then the modified C at 234523 will be retained (Nmod=1).}

\item{parallel}{Logical. If TRUE, automatically detect your system condition and
decoding will use parallel threads via BiocParallel::.
If FALSE (default), decoding is done serially.}

\item{BPPARAM}{A \code{BiocParallelParam} object controlling parallel execution.
This value will automatically set when parallel is \code{TRUE}, also able to set as manually.}
}
\value{
A \code{GRanges} of the same length as \code{ref_gr}, with four
  additional metadata columns:
  \describe{
    \item{ONT_Cm}{comma-delimited read‐indices called modified}
    \item{ONT_C}{comma-delimited read‐indices covering but unmodified}
    \item{readsM}{integer count of modified reads per site}
    \item{readsN}{integer count of same‐strand reads covering each site}
  }
}
\description{
\code{readONTbam()} takes an indexed Nanopore BAM file with MM/ML tags,
decodes each read’s per-C modification probabilities, and overlays
them on a \code{GRanges} of candidate cytosine sites.
It returns a copy of \code{ref_gr} augmented with:
\itemize{
  \item \code{ONT_Cm} — comma-delimited read‐indices called “modified”
  \item \code{ONT_C}  — comma-delimited read‐indices covering but _not_ modified
  \item \code{readsM} — count of modified reads at each site
  \item \code{readsN} — total same-strand coverage at each site
}

You can either supply your own \code{ref_gr} (e.g.\ from \code{selectCytosine()})
or leave it \code{NULL} and pass \code{context}, \code{chr}, \code{region}
to build \code{ref_gr} on the fly.
}
\details{
This function read and annotate ONT MM/ML tags against a cytosine reference
}
\examples{
\dontrun{
library(DMRcaller)
library(BSgenome.Hsapiens.UCSC.hg38)

# set the bam file directory
bam_path <- system.file("extdata", "scanBamChr1Random5.bam", package="DMRcaller")

# read ONTbam file (chromosome 1 only) in CG context with BSgenome.Hsapiens.UCSC.hg38
ONTSampleGRanges <- readONTbam(bamfile = bam_path, ref_gr = NULL, modif = "C+m?",
                         prob_thresh = 0.50,genome = BSgenome.Hsapiens.UCSC.hg38,
                         context = "CG", chr = "chr1", region = NULL,
                         synonymous = FALSE, parallel = FALSE, BPPARAM = NULL)
}

}
\seealso{
\code{\link{selectCytosine}}, \code{\link{computeDMRs}},
\code{\link{computePMDs}}, \code{\link{computeCoMethylation}},
\code{\link{filterVMRsONT}}, \code{\link{ontSampleGRangesList}},
\code{\link{scanBamChr1Random5}}
}
\author{
Nicolae Radu Zabet and Young Jun Kim
}
