% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coMethylation.R
\name{computeCoMethylation}
\alias{computeCoMethylation}
\title{Compute pairwise co-methylation statistics for cytosine sites within regions}
\usage{
computeCoMethylation(
  methylationData,
  regions,
  minDistance = 150,
  maxDistance = 1000,
  minCoverage = 4,
  pValueThreshold = 0.01,
  alternative = "two.sided",
  test = "fisher",
  parallel = FALSE,
  BPPARAM = NULL
)
}
\arguments{
\item{methylationData}{A \code{GRanges} object containing cytosine sites, annotated with
per-site ONT methylation calls (columns \code{ONT_Cm}, \code{ONT_C}, \code{readsN}, etc).}

\item{regions}{A \code{GRanges} object with list including genomic context such as gene and/or transposable
elements coordinates which possibly have DMRs, VMRs or PMDs.}

\item{minDistance}{Minimum distance (in bp) between two cytosines to consider for co-methylation (default: 150).}

\item{maxDistance}{Maximum distance (in bp) between two cytosines to consider (default: 1000).}

\item{minCoverage}{Minimum read coverage required for both cytosines in a pair (default: 4).}

\item{pValueThreshold}{FDR-adjusted p-value threshold for reporting significant co-methylation (default: 0.01).}

\item{alternative}{indicates the alternative hypothesis and must be one of
\code{"two.sided"}, \code{"greater"} or \code{"less"}. You can specify just
the initial letter. Only used in the 2 by 2 case. This is used only for
Fisher's test.}

\item{test}{Statistical test to use for co-methylation (\code{"fisher"}
for Fisher's exact [default], or \code{"permutation"} for chi-squared).
NOTE: highly recommended to do parallel when use permutation test.}

\item{parallel}{Logical; run in parallel if \code{TRUE}.}

\item{BPPARAM}{A \code{BiocParallelParam} object controlling parallel execution.
This value will automatically set when parallel is \code{TRUE}, also able to set as manually.}
}
\value{
A list of length equal to \code{regions}, where each entry is a \code{GInteractions} object
  of significant cytosine pairs (by strand), annotated with:
  \describe{
    \item{C1_C2}{number of reads methylated at both cytosines}
    \item{C1_only}{number methylated at only first cytosine}
    \item{C2_only}{number methylated at only second cytosine}
    \item{neither}{number methylated at neither cytosines}
    \item{strand}{The DNA strand ("+" or "-") on which the two CpGs reside.}
    \item{genomic_position}{The original region (from \code{regions}) containing
    this cytosines pair, formatted in UCSC or IGV style, e.g. \code{"chr1:1522971-1523970"}.}
    \item{p.value}{FDR-adjusted p-value for co-methylation association}
  }
}
\description{
\code{computeCoMethylation()} calculates pairwise co-methylation between all cytosine sites
within each given region, using ONT methylation calls annotated to each site.
For each pair of cytosines within the same strand and PMD, it builds a 2x2 contingency table
reflecting the overlap state of reads (both methylated, only one methylated, or neither),
performs a statistical test (Fisher's exact by default), and reports FDR-adjusted p-values.
}
\details{
Compute Co-Methylation Between CpG Sites Within regions


Pairwise tests are performed separately for each strand (+ and -) within each region.
FDR correction is performed for all pairs within each region and strand.
}
\examples{
\dontrun{
# load the ONT methylation data and PMD data
data(ont_gr_GM18870_chr1_PMD_bins_1k)
data(ont_gr_GM18870_chr1_sorted_bins_1k)

# compute the co-methylations with Fisher's exact test
coMetylationFisher <- computeCoMethylation(
  ont_gr_GM18870_chr1_sorted_bins_1k,
  regions = ont_gr_GM18870_chr1_PMD_bins_1k[1:4],
  minDistance = 150,
  maxDistance = 1000,
  minCoverage = 4,
  pValueThreshold = 0.01,
  test = "fisher",
  parallel = FALSE)

# compute the co-methylations with Permuation test
coMetylationPermutation <- computeCoMethylation(
  ont_gr_GM18870_chr1_sorted_bins_1k,
  regions = ont_gr_GM18870_chr1_PMD_bins_1k[1:4],
  minDistance = 150,
  maxDistance = 1000,
  minCoverage = 4,
  pValueThreshold = 0.01,
  test = "permutation",
  parallel = FALSE) # highly recommended to set as TRUE
}

}
\seealso{
\code{\link{readONTbam}}, \code{\link{computePMDs}},
\code{\link{ontSampleGRangesList}}
}
\author{
Nicolae Radu Zabet and Young Jun Kim
}
