#' DEP: A package for Differential Enrichment analysis of Proteomics data.
#'
#' This package provides an integrated analysis workflow for robust and
#' reproducible analysis of mass spectrometry proteomics data for
#' differential protein expression or differential enrichment.
#' It requires tabular input (e.g. txt files) as generated by quantitative
#' analysis softwares of raw mass spectrometry data, such as
#' \href{http://www.nature.com/nbt/journal/v26/n12/full/nbt.1511.html}{MaxQuant} or
#' \href{http://www.nature.com/nprot/journal/v10/n10/full/nprot.2015.101.html}{IsobarQuant}.
#' Functions are provided for data preparation, filtering,
#' variance normalization and imputation of missing values,
#' as well as statistical testing of differentially enriched / expressed proteins.
#' It also includes tools to check intermediate steps in the workflow,
#' such as normalization and missing values imputation.
#' Finally, visualization tools are provided to explore the results,
#' including heatmap, volcano plot and barplot representations.
#' For scientists with limited experience in R, the package also entails
#' wrapper functions that entail the complete analysis workflow and generate
#' a report. Even easier to use are the interactive Shiny apps that
#' are provided by the package.
#'
#' @section Shiny apps:
#' \itemize{
#'   \item \code{\link{run_app}}: Shiny apps for interactive analysis.
#' }
#'
#' @section Workflow functions:
#' \itemize{
#'   \item \code{\link{LFQ}}:
#'   Label-free quantification (LFQ) workflow wrapper.
#'   \item \code{\link{TMT}}:
#'   Tandem-mass-tags (TMT) workflow wrapper.
#'   \item \code{\link{report}}:
#'   Create a rmarkdown report wrapper.
#' }
#'
#' @section Wrapper functions:
#' \itemize{
#'   \item \code{\link{import_MaxQuant}}:
#'   Import data from MaxQuant into a SummarizedExperiment object.
#'   \item \code{\link{import_IsobarQuant}}:
#'   Import data from IsobarQuant  into a SummarizedExperiment object.
#'   \item \code{\link{process}}:
#'   Perform filtering, normalization and imputation on protein data.
#'   \item \code{\link{analyze_dep}}:
#'   Differential protein expression analysis.
#'   \item \code{\link{plot_all}}:
#'   Visualize the results in different types of plots.
#' }
#'
#' @section Main functions:
#' \itemize{
#'   \item \code{\link{make_unique}}:
#'   Generate unique names.
#'   \item \code{\link{make_se_parse}}:
#'   Turn data.frame into SummarizedExperiment by parsing column names.
#'   \item \code{\link{make_se}}:
#'   Turn data.frame into SummarizedExperiment using an experimental design.
#'   \item \code{\link{filter_proteins}}:
#'   Filter proteins based on missing values.
#'   \item \code{\link{normalize_vsn}}:
#'   Normalize data using vsn.
#'   \item \code{\link{impute}}:
#'   Impute missing values.
#'   \item \code{\link{test_diff}}:
#'   Differential enrichment analysis.
#'   \item \code{\link{add_rejections}}:
#'   Mark significant proteins.
#'   \item \code{\link{get_results}}:
#'   Generate a results table.
#' }
#'
#' @section Visualization functions:
#' \itemize{
#'   \item \code{\link{plot_single}}:
#'   Barplot for a protein of interest.
#'   \item \code{\link{plot_volcano}}:
#'   Volcano plot for a specified contrast.
#'   \item \code{\link{plot_heatmap}}:
#'   Heatmap of all significant proteins.
#'   \item \code{\link{plot_normalization}}:
#'   Boxplots to inspect normalization.
#'   \item \code{\link{plot_detect}}:
#'   Density and CumSum plots of proteins
#'   with and without missing values.
#'   \item \code{\link{plot_imputation}}:
#'   Density plots to inspect imputation.
#'   \item \code{\link{plot_missval}}:
#'   Heatmap to inspect missing values.
#'   \item \code{\link{plot_numbers}}:
#'   Barplot of proteins identified.
#'   \item \code{\link{plot_frequency}}:
#'   Barplot of protein identification overlap between conditions.
#'   \item \code{\link{plot_coverage}}:
#'   Barplot of the protein coverage in conditions.
#'   \item \code{\link{plot_pca}}:
#'   PCA plot of top variable proteins.
#'   \item \code{\link{plot_cor}}:
#'   Plot correlation matrix.
#'   \item \code{\link{plot_cor}}:
#'   Plot Gower's distance matrix.
#'   \item \code{\link{plot_p_hist}}:
#'   P value histogram.
#'   \item \code{\link{plot_cond_freq}}:
#'   Barplot of the number of significant conditions per protein.
#'   \item \code{\link{plot_cond_overlap}}:
#'   Barplot of the number of proteins for overlapping conditions.
#'   \item \code{\link{plot_cond}}:
#'   Barplot of the frequency of significant conditions per protein
#'   and the overlap in proteins between conditions.
#' }
#'
#' @section Gene Set Enrichment Analysis functions:
#' \itemize{
#'   \item \code{\link{test_gsea}}:
#'   Gene Set Enrichment Analysis using enrichR.
#'   \item \code{\link{plot_gsea}}:
#'   Barplot of enriched gene sets.
#' }
#'
#' @section Additional functions:
#' \itemize{
#'   \item \code{\link{get_df_wide}}:
#'   Generate a wide data.frame from a SummarizedExperiment.
#'   \item \code{\link{get_df_long}}:
#'   Generate a long data.frame from a SummarizedExperiment.
#'   \item \code{\link{se2msn}}:
#'   SummarizedExperiment object to MSnSet object conversion.
#'   \item \code{\link{filter_missval}}:
#'   Filter on missing values.
#'   \item \code{\link{manual_impute}}:
#'   Imputation by random draws from a manually defined distribution.
#'   \item \code{\link{get_prefix}}:
#'   Obtain the longest common prefix.
#'   \item \code{\link{get_suffix}}:
#'   Obtain the longest common suffix.
#' }
#'
#' @section Example data:
#' \itemize{
#'   \item \code{\link{UbiLength}}:
#'   Ubiquitin interactors of different linear ubiquitin lengths
#'   (UbIA-MS dataset) (Zhang, Smits, van Tilburg et al. Mol. Cell 2017).
#'   \item \code{\link{UbiLength_ExpDesign}}:
#'   Experimental design of the UbiLength dataset.
#'   \item \code{\link{DiUbi}}:
#'   Ubiquitin interactors for different diubiquitin-linkages
#'   (UbIA-MS dataset) (Zhang, Smits, van Tilburg et al. Mol. Cell 2017).
#'   \item \code{\link{DiUbi_ExpDesign}}:
#'   Experimental design of the DiUbi dataset.
#' }
#'
#' @docType package
#' @name DEP
#'
#'
#' @import ggplot2 dplyr SummarizedExperiment limma
#' @import ComplexHeatmap grid assertthat imputeLCMD
#' @import shinydashboard
#' @rawNamespace import(MSnbase, except = c(combine, exprs))
#' @importFrom shiny runApp
#' @importFrom DT dataTableOutput renderDataTable
#' @importFrom tidyr unite gather spread separate
#' @importFrom tibble rownames_to_column column_to_rownames
#' @importFrom purrr map_df
#' @importFrom readr parse_factor
#' @importFrom stats median model.matrix rnorm sd cor prcomp formula terms.formula
#' @importFrom cluster daisy
#'
#'
NULL
