## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL 
    ## Related to 
    ## https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE----------------
## Bib setup
library("RefManageR")

## Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitr = citation("knitr")[1],
    RefManageR = citation("RefManageR")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    CalibraCurve = citation("CalibraCurve")[1],
    ggplot2 = citation("ggplot2")[1],
    msqc1 = citation("msqc1")[1]
    
)

BibOptions(max.names = 2, bib.style = "authoryear", style = "citation")


## ----"install", eval = FALSE--------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#         install.packages("BiocManager")
#     }
# 
# BiocManager::install("CalibraCurve")

## ----"citation"---------------------------------------------------------------
## Citation info
citation("CalibraCurve")

## ----"start", message=FALSE---------------------------------------------------
library("CalibraCurve")

file <- system.file("extdata", "MSQC1_xlsx", "GGPFSDSYR_QExactive_y5.xlsx", 
                    package = "CalibraCurve")

D <- readDataTable(dataPath = file, 
                    fileType = "xlsx", 
                    concCol = 16, # column "amount" containing concentrations
                    measCol = 12, # column "Area" containing measurements
                    naStrings = c("NA", "NaN", "Filtered", "#NV"), 
                    sheet = 1)
print(head(D))


## ----"start2", message=FALSE--------------------------------------------------
library("CalibraCurve")

folder <- system.file("extdata", "MSQC1_xlsx", package = "CalibraCurve")

D <- readMultipleTables(dataFolder = folder, 
                        fileType = "xlsx", 
                        concCol = 16, 
                        measCol = 12) 
print(D)

str(D)


## ----"start_SE", message=FALSE------------------------------------------------
library("CalibraCurve")

file <- system.file("extdata", "MSQC1", "msqc1_dil_GGPFSDSYR.rds", 
                    package = "CalibraCurve")

D <- readDataSE(file, concColName = "amount_fmol", substColName = "Substance")

print(D)

str(D)

## ----"start_SE2", message=FALSE-----------------------------------------------
library("CalibraCurve")

file <- system.file("extdata", "MSQC1", "msqc1_dil_GGPFSDSYR.rds", 
                    package = "CalibraCurve")

SE_data <- readRDS(file)

D <- readDataSE(rawDataSE = SE_data, 
                concColName = "amount_fmol", 
                substColName = "Substance")

print(D)

str(D)

## ----"apply_CC", message=FALSE------------------------------------------------
RES <- CalibraCurve(D)



## ----"plot_SE", message=FALSE-------------------------------------------------
print(RES$plot_CC_list[[4]])

## ----"plot_RF", message=FALSE-------------------------------------------------
print(RES$plot_RF_list[[4]])

## ----"print_table_conc_levels", message=FALSE---------------------------------
print(RES$RES[[4]]$result_table_conc_levels)

## ----"print_table_obs", message=FALSE-----------------------------------------
print(RES$RES[[4]]$result_table_obs)

## ----"print_table_summary", message=FALSE-------------------------------------
print(RES$summary_tab)

## ----"prediction", message=FALSE----------------------------------------------
newdata <- c(1000000, #1e6
            10000000, #1e7
            100000000) # 1e8


CC_res = RES$RES[[4]]

predictConcentration(CC_res = RES$RES[[4]], newdata = newdata) 

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----Biblio, results = "asis", echo = FALSE, warning = FALSE, message = FALSE-----------------------------------------
## Print bibliography
PrintBibliography(bib, .opts = list(hyperlink = "to.doc", style = "html"))

