% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getOfftargetWithBulge.R
\name{getOfftargetWithBulge}
\alias{getOfftargetWithBulge}
\title{Identify off-targets with bulges for target-specific gRNAs designed for 
CRISPR-Cas9 systems.}
\usage{
getOfftargetWithBulge(
  gRNA_PAM = NULL,
  output_csv_name = NULL,
  PAM.size = 3,
  PAM.pattern = "NNG$|NGN$",
  PAM.location = c("3prime", "5prime"),
  max.mismatch = 3,
  DNA_bulge = 2,
  RNA_bulge = 2,
  BSgenomeName = NULL,
  genomeSeqFile = NULL,
  chromToSearch = "all",
  chromToExclude = NULL,
  cas_offinder_version = c("2.4.1", "3.0.0b3")
)
}
\arguments{
\item{gRNA_PAM}{A `DNAStringSet` object returned by `findgRNA()` that contains
gRNA plus PAM sequences. Alternatively, you can supply the `list` object 
returned by the `offTargetAnalysis()` function.}

\item{output_csv_name}{A string specifying the output CSV file name. Defaults
to `NULL`, meaning that the output will be printed to the console.}

\item{PAM.size}{See `offTargetAnalysis()`.}

\item{PAM.pattern}{See `offTargetAnalysis()`.}

\item{PAM.location}{See `offTargetAnalysis()`.}

\item{max.mismatch}{See `offTargetAnalysis()`.}

\item{DNA_bulge}{The maximum size of DNA bulges, specified in nucleotides. 
Defaults to 2.}

\item{RNA_bulge}{The maximum size of RNA bulges, specified in nucleotides. 
Defaults to 2.}

\item{BSgenomeName}{See `offTargetAnalysis()`. Alternatively, use 
`genomeSeqFile` to specify the file path to custom genome fasta file. Note,
`genomeSeqFile` overwrites `BSgenomeName` if both set.}

\item{genomeSeqFile}{If you are using a custom genome, specify the file
path to the FASTA file using `genomeSeqFile`.}

\item{chromToSearch}{See `offTargetAnalysis()`.}

\item{chromToExclude}{See `offTargetAnalysis()`.}

\item{cas_offinder_version}{The version of "Cas-OFFinder" to use. 
Currently supported versions are "2.4.1" and "3.0.0b3". Defaults to "2.4.1".}
}
\value{
If `output_csv_name` is not set, the function returns a data frame
containing the output generated by `Cas-OFFinder`. Otherwise, it saves the
data frame to the CSV file specified by `output_csv_name`. When
`cas_offinder_version == "2.4.1"`, the following columns will be included: "bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size". For `cas_offinder_version == "3.0.0b3"`, the included columns will be: "gRNA_id", "bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size".
}
\description{
This function extends the off-targets identified by offTargetAnalysis() by 
detecting off-targets that contain bulges. In gRNA design, "bulges" refer to
insertions ("RNA bulges") or deletions ("DNA bulges") in the gRNA sequence 
relative to the target DNA sequence. Bulges can affect the binding affinity 
and specificity of the gRNA to its target. The function wraps around 
[`Cas-OFFinder`](http://www.rgenome.net/cas-offinder/) internally.
}
\examples{
# Example with `DNAStringSet` as input
if (interactive()) {
  library(CRISPRseek)
  library(BSgenome.Hsapiens.UCSC.hg19)

  gRNA_PAM <- findgRNAs(inputFilePath = system.file("extdata", 
                                                    "inputseq.fa", 
                                                     package = "CRISPRseek"),
                        pairOutputFile = "testpairedgRNAs.xls",
                        findPairedgRNAOnly = TRUE)
  df <- getOfftargetWithBulge(gRNA_PAM, PAM.pattern = "NNG$|NGN$",
                             DNA_bulge = 2, RNA_bulge = 2,
                             BSgenomeName = Hsapiens, chromToSearch = "chrX")

 # Example with `list` output from `offTargetAnalysis` as input
 library(TxDb.Hsapiens.UCSC.hg19.knownGene)
 library(org.Hs.eg.db)

 inputFilePath <- system.file("extdata", "inputseq.fa", package = "CRISPRseek")
 REpatternFile <- system.file("extdata", "NEBenzymes.fa", package = "CRISPRseek")
 res <- offTargetAnalysis(inputFilePath, 
                          findgRNAsWithREcutOnly = TRUE,
                          REpatternFile = REpatternFile, 
                          findPairedgRNAOnly = FALSE, 
                          annotatePaired = FALSE,
                          BSgenomeName = Hsapiens, 
                          chromToSearch = "chrX",
                          txdb = TxDb.Hsapiens.UCSC.hg19.knownGene,
                          orgAnn = org.Hs.egSYMBOL, max.mismatch = 1,
                          outputDir = tempdir(),
                          overwrite = TRUE)
 df <- getOfftargetWithBulge(res, PAM.pattern = "NNG$|NGN$",
                             DNA_bulge = 2, 
                             RNA_bulge = 2, 
                             BSgenomeName = Hsapiens, 
                             chromToSearch = "chrX") 
 }
}
\references{
1. Sangsu Bae, Jeongbin Park, Jin-Soo Kim, Cas-OFFinder: a fast and versatile algorithm that searches for potential off-target sites of Cas9 RNA-guided endonucleases, Bioinformatics, Volume 30, Issue 10, May 2014, Pages 1473–1475, https://doi.org/10.1093/bioinformatics/btu048
}
\seealso{
`offTargetAnalysis()` for off-targets analysis,
`Cas-OFFinder` (https://github.com/snugel/cas-offinder) for more on output format.
}
\author{
Kai Hu
}
