#' @title Bar plot of missing values per lines using highcharter.
#'
#' @description
#'
#' This method plots a bar plot which represents the distribution of the
#' number of missing values (NA) per lines (ie proteins).
#'
#' * `wrapper_pca()`
#' * `plotPCA_Eigen_hc()`: plots the eigen values of PCA with the highcharts
#'    library
#' * `plotPCA_Eigen()`: plots the eigen values of PCA
#' * `plotPCA_Var()`
#' * `plotPCA_Ind()`
#'
#' @param id A `character(1)` which is the id of the shiny module.
#' @param dataIn An instance of the class `MultiAssayExperiment`.
#' @param i An integer which is the index of the assay in the param obj
#' @param var.scaling The dimensions to plot
#' @param ncp A `integer(1)` which represents the umber of dimensions kept in
#' the results.
#' @param res.pca Result of FactoMineR::PCA
#' @param chosen.axes The dimensions to plot
#'
#' @author Samuel Wieczorek, Enora Fremy
#'
#' @name ds-pca
#'
#'
#' @author Samuel Wieczorek, Enora Fremy
#'
#' @examples
#' if (interactive()) {
#'     data(vdata)
#'     library(shiny)
#'     library(dplyr)
#'     library(highcharter)
#'     # Replace missing values for the example
#'     sel <- is.na(SummarizedExperiment::assay(vdata, 1))
#'     SummarizedExperiment::assay(vdata[[1]])[sel] <- 0
#'     SummarizedExperiment::assay(vdata[[1]])[1, 1] <- NA
#'     omXplore_pca(vdata, 1)
#' }
#'
NULL



#' @importFrom shiny shinyApp reactive NS tagList tabsetPanel tabPanel fluidRow
#' column uiOutput radioButtons reactive moduleServer reactiveValues observeEvent
#' renderUI req selectInput isolate uiOutput tagList checkboxInput fluidPage div
#'  p numericInput observe plotOutput renderImage renderPlot selectizeInput
#' sliderInput textInput updateSelectInput updateSelectizeInput wellPanel
#' withProgress h3 br actionButton addResourcePath h4 helpText imageOutput
#' @importFrom shinyjs useShinyjs hidden toggle
#' @importFrom RColorBrewer brewer.pal
#' @importFrom highcharter renderHighchart
#' @importFrom shinyjs useShinyjs hidden toggle
#'
#' @rdname ds-pca
#' @export
#' @return NA
#'
omXplore_pca_ui <- function(id) {
    ns <- NS(id)
    tagList(
        shinyjs::useShinyjs(),
        shinyjs::hidden(div(
            id = ns("badFormatMsg"),
            h3(globals()$bad_format_txt)
        )),
        uiOutput(ns("WarningNA_PCA")),
        uiOutput(ns("pcaOptions")),
        shinyjs::hidden(checkboxInput(ns("gramschmidt_PCA"), "gramschmidt in Nipals", value = TRUE)),
        shinyjs::hidden(checkboxInput(ns("PCA_varScale"), "VarScale in FactoMineR", value = TRUE)),
        uiOutput(ns("pcaPlots"))
    )
}


#' @importFrom shiny shinyApp reactive NS tagList tabsetPanel tabPanel fluidRow
#' column uiOutput radioButtons reactive moduleServer reactiveValues observeEvent
#' renderUI req selectInput isolate uiOutput tagList checkboxInput fluidPage div
#'  p numericInput observe plotOutput renderImage renderPlot selectizeInput
#' sliderInput textInput updateSelectInput updateSelectizeInput wellPanel
#' withProgress h3 br actionButton addResourcePath h4 helpText imageOutput
#' @importFrom shinyjs useShinyjs hidden toggle
#' @importFrom RColorBrewer brewer.pal
#' @importFrom highcharter renderHighchart
#' @importFrom shinyjs useShinyjs hidden toggle
#' @importFrom SummarizedExperiment assay
#'
#' @rdname ds-pca
#'
#' @import factoextra
#'
#' @export
#' @return NA
#'
omXplore_pca_server <- function(
        id,
        dataIn,
        i) {
    moduleServer(id, function(input, output, session) {
        ns <- session$ns

        rv.pca <- reactiveValues(
            data = NULL,
            PCA_axes = NULL,
            res.pca = NULL,
            PCA_varScale = TRUE,
            gramschmidt_PCA = TRUE,
            approach_PCA = "FM"
        )


        observe(
            {
                is.mae <- inherits(dataIn(), "MultiAssayExperiment")
                stopifnot(is.mae)

                rv.pca$data <- SummarizedExperiment::assay(dataIn(), i())

                shinyjs::toggle("badFormatMsg", condition = !is.mae)
            },
            priority = 1000
        )

        output$WarningNA_PCA <- renderUI({
            # req(rv.pca$data)
            # req(length(which(is.na(rv.pca$data))) > 0)

            rule <- !is.null(rv.pca$data) && length(which(is.na(rv.pca$data))) > 0
            req(rule)
            if (rv.pca$approach_PCA == "FM") {
                tagList(
                    tags$p(
                        style = "color:red;font-size: 20px",
                        "Warning: As your dataset contains missing values,
            the PCA cannot be computed. Please impute them first or use Nipals."
                    )
                )
            }
        })


        output$pcaOptions <- renderUI({
            req(rv.pca$data)
            print(length(rv.pca$data))
            # req(length(which(is.na(rv.pca$data))) == 0)
            tagList(
                tags$div(
                    tags$div(
                        style = "display:inline-block;
                             vertical-align: middle; padding-right: 20px;",
                        numericInput(ns("pca_axe1"), "Dimension 1",
                            min = 1,
                            max = Compute_PCA_dim(),
                            value = 1,
                            width = "100px"
                        )
                    ),
                    tags$div(
                        style = "display:inline-block; vertical-align: middle;",
                        numericInput(ns("pca_axe2"), "Dimension 2",
                            min = 1,
                            max = Compute_PCA_dim(),
                            value = 2,
                            width = "100px"
                        )
                    ),
                    tags$div(
                        selectInput(ns("approach_PCA"),
                            label = "Approach used for PCA",
                            choices = c("FactoMineR" = "FM", "Nipals" = "NIPALS"),
                            width = "150px"
                        )
                    )
                )
            )
        })

        observeEvent(c(input$pca_axe1, input$pca_axe2), {
            rv.pca$PCA_axes <- c(input$pca_axe1, input$pca_axe2)
        })

        observeEvent(req(input$PCA_varScale), {
            rv.pca$PCA_varScale <- input$PCA_varScale
        })

        observeEvent(input$gramschmidt_PCA, {
            rv.pca$gramschmidt_PCA <- input$gramschmidt_PCA
        })
        observeEvent(req(input$approach_PCA), {
            rv.pca$approach_PCA <- input$approach_PCA

            shinyjs::toggle("gramschmidt_PCA", condition = rv.pca$approach_PCA == "NIPALS")
        })

        observeEvent(input$PCA_varScale, {
            rv.pca$PCA_varScale <- input$PCA_varScale
        })
        observeEvent(req(input$approach_PCA), {
            rv.pca$approach_PCA <- input$approach_PCA
            shinyjs::toggle("PCA_varScale", condition = rv.pca$approach_PCA == "FM")
        })

        observe({
            rule1 <- rv.pca$approach_PCA == "FM" && length(which(is.na(rv.pca$data))) == 0
            rule2 <- rv.pca$approach_PCA == "NIPALS"
            req(rule1 || rule2)

            rv.pca$res.pca <- wrapper_pca(
                qdata = SummarizedExperiment::assay(dataIn(), i()),
                group = get_group(dataIn()),
                var.scaling = rv.pca$PCA_varScale,
                ncp = Compute_PCA_dim(),
                approach = rv.pca$approach_PCA,
                gramschmidt = rv.pca$gramschmidt_PCA
            )
        })

        output$pcaPlots <- renderUI({
            req(rv.pca$data)
            req(rv.pca$res.pca$var$coord)

            rule1 <- rv.pca$approach_PCA == "FM" && length(which(is.na(rv.pca$data))) == 0
            rule2 <- rv.pca$approach_PCA == "NIPALS"
            req(rule1 || rule2)

            tagList(
                plotOutput(ns("pcaPlotVar")),
                plotOutput(ns("pcaPlotInd")),
                formatDT_ui(ns("PCAvarCoord")),
                highcharter::highchartOutput(ns("pcaPlotEigen"))
            )
        })

        observe({
            df <- as.data.frame(rv.pca$res.pca$var$coord)
            formatDT_server("PCAvarCoord",
                data = reactive({
                    round(df, digits = 2)
                }),
                showRownames = TRUE
            )
        })

        output$pcaPlotVar <- renderPlot({
            req(c(rv.pca$PCA_axes, rv.pca$res.pca))

            withProgress(message = "Making plot", value = 100, {
                factoextra::fviz_pca_var(rv.pca$res.pca,
                    axes = rv.pca$PCA_axes,
                    col.var = "cos2",
                    gradient.cols = c("#00AFBB", "#E7B800", "#FC4E07"),
                    repel = TRUE
                )
            })
        })

        output$pcaPlotInd <- renderPlot({
            req(c(rv.pca$PCA_axes, rv.pca$res.pca))

            withProgress(message = "Making plot", value = 100, {
                factoextra::fviz_pca_ind(rv.pca$res.pca,
                    axes = rv.pca$PCA_axes,
                    geom = "point"
                )
            })
        })


        output$pcaPlotEigen <- highcharter::renderHighchart({
            req(rv.pca$res.pca)

            withProgress(message = "Making plot", value = 100, {
                plotPCA_Eigen(rv.pca$res.pca)
            })
        })


        Compute_PCA_dim <- reactive({
            req(rv.pca$data)
            nmax <- 12 # ncp should not be greater than...
            # for info, ncp = number of components or dimensions in PCA results

            y <- rv.pca$data
            nprot <- dim(y)[1]
            n <- dim(y)[2] # If too big, take the number of conditions.

            if (n > nmax) {
                n <- length(unique(get_group(dataIn)))
            }

            ncp <- min(n, nmax)
            ncp
        })
    })
}


#' @export
#' @rdname ds-pca
#' @return A shiny app
#'
omXplore_pca <- function(dataIn, i) {
    stopifnot(inherits(dataIn, "MultiAssayExperiment"))

    ui <- omXplore_pca_ui("plot")

    server <- function(input, output, session) {
        omXplore_pca_server("plot",
            dataIn = reactive({
                dataIn
            }),
            i = reactive({
                i
            })
        )
    }
    shinyApp(ui = ui, server = server)
}
