% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peaks-functions.R
\name{joinPeaks}
\alias{joinPeaks}
\alias{joinPeaksGnps}
\alias{joinPeaksNone}
\title{Join (map) peaks of two spectra}
\usage{
joinPeaks(x, y, type = "outer", tolerance = 0, ppm = 10, ...)

joinPeaksGnps(
  x,
  y,
  xPrecursorMz = NA_real_,
  yPrecursorMz = NA_real_,
  tolerance = 0,
  ppm = 0,
  type = "outer",
  ...
)

joinPeaksNone(x, y, ...)
}
\arguments{
\item{x}{\code{matrix} with two columns \code{"mz"} and \code{"intensity"} containing the
m/z and intensity values of the mass peaks of a spectrum.}

\item{y}{\code{matrix} with two columns \code{"mz"} and \code{"intensity"} containing the
m/z and intensity values of the mass peaks of a spectrum.}

\item{type}{For \code{joinPeaks()} and \code{joinPeaksGnps()}: \code{character(1)}
specifying the type of join that should be performed. See function
description for details.}

\item{tolerance}{\code{numeric(1)} defining a constant maximal accepted difference
between m/z values of peaks from the two spectra to be matched/mapped.}

\item{ppm}{\code{numeric(1)} defining a relative, m/z-dependent, maximal accepted
difference between m/z values of peaks from the two spectra to be
matched/mapped.}

\item{...}{optional parameters passed to the \code{\link[MsCoreUtils:matching]{MsCoreUtils::join()}} function.}

\item{xPrecursorMz}{for \code{joinPeaksGnps()}: \code{numeric(1)} with the precursor
m/z of the spectrum \code{x}.}

\item{yPrecursorMz}{for \code{joinPeaksGnps()}: \code{numeric(1)} with the precursor
m/z of the spectrum \code{y}.}
}
\value{
All functions return a \code{list} of elements \code{"x"} and \code{"y"} each being a two
column matrix with m/z (first column) and intensity values (second column).
The two matrices contain the matched peaks between input matrices \code{x} and \code{y}
and hence have the same number of rows. Peaks present in \code{x} but not in the
\code{y} input matrix have m/z and intensity values of \code{NA} in the result matrix
for \code{y} (and \emph{vice versa}).
}
\description{
These functions map peaks from two spectra with each other if the difference
between their m/z values is smaller than defined with parameters \code{tolerance}
and \code{ppm}. All functions take two matrices
\itemize{
\item \code{joinPeaks()}: maps peaks from two spectra allowing to specify the type of
\emph{join} that should be performed: \code{type = "outer"} each peak in \code{x} will be
matched with each peak in \code{y}, for peaks that do not match any peak in the
other spectra an \code{NA} intensity is returned. With \code{type = "left"} all peaks
from the left spectrum (\code{x}) will be matched with peaks in \code{y}. Peaks in
\code{y} that do not match any peak in \code{x} are omitted. \code{type = "right"} is the
same as \code{type = "left"} only for \code{y}. Only peaks that can be matched
between \code{x} and \code{y} are returned by \code{type = "inner"}, i.e. only
peaks present in both spectra are reported.
\item \code{joinPeaksGnps()}: matches/maps peaks between spectra with the same
approach used in GNPS: peaks are considered matching if a) the
difference in their m/z values is smaller than defined by \code{tolerance}
and \code{ppm} (this is the same as \code{joinPeaks}) \strong{and} b) the difference of
their m/z \emph{adjusted} for the difference of the spectras' precursor is
smaller than defined by \code{tolerance} and \code{ppm}. Based on this definition,
peaks in \code{x} can match up to two peaks in \code{y} hence peaks in the returned
matrices might be reported multiple times. Note that if one of
\code{xPrecursorMz} or \code{yPrecursorMz} are \code{NA} or if both are the same, the
results are the same as with \code{\link[=joinPeaks]{joinPeaks()}}. To calculate GNPS similarity
scores, \code{\link[MsCoreUtils:gnps]{MsCoreUtils::gnps()}} should be called on the aligned peak
matrices (i.e. \code{compareSpectra} should be called with
\code{MAPFUN = joinPeaksGnps} and \code{FUN = MsCoreUtils::gnps}).
\item \code{joinPeaksNone()}: does not perform any peak matching but simply returns
the peak matrices in a \code{list}. This function should be used with the
\code{MAPFUN} parameter of \code{\link[=compareSpectra]{compareSpectra()}} if the spectra similarity
function used (parameter \code{FUN} of \code{compareSpectra()}) performs its own
peak matching and does hence not expect matched peak matrices as an input.
}
}
\section{Implementation notes}{


A mapping function must take two numeric matrices \code{x} and \code{y} as input and
must return \code{list} with two elements named \code{"x"} and \code{"y"} that represent
the aligned input matrices. The function should also have \code{...} in its
definition. Parameters \code{ppm} and \code{tolerance} are suggested but not required.
}

\examples{

x <- cbind(c(31.34, 50.14, 60.3, 120.9, 230, 514.13, 874.1),
    1:7)
y <- cbind(c(12, 31.35, 70.3, 120.9 + ppm(120.9, 5),
    230 + ppm(230, 10), 315, 514.14, 901, 1202),
    1:9)

## No peaks with identical m/z
joinPeaks(x, y, ppm = 0, type = "inner")

## With ppm 10 two peaks are overlapping
joinPeaks(x, y, ppm = 10, type = "inner")

## Outer join: contain all peaks from x and y
joinPeaks(x, y, ppm = 10, type = "outer")

## Left join: keep all peaks from x and those from y that match
joinPeaks(x, y, ppm = 10, type = "left")

## Right join: keep all peaks from y and those from x that match. Using
## a constant tolerance of 0.01
joinPeaks(x, y, tolerance = 0.01, type = "right")

## GNPS-like peak matching

## Define spectra
x <- cbind(mz = c(10, 36, 63, 91, 93), intensity = c(14, 15, 999, 650, 1))
y <- cbind(mz = c(10, 12, 50, 63, 105), intensity = c(35, 5, 16, 999, 450))
## The precursor m/z
pmz_x <- 91
pmz_y <- 105

## Plain joinPeaks identifies only 2 matching peaks: 1 and 5
joinPeaks(x, y)

## joinPeaksGnps finds 4 matches
joinPeaksGnps(x, y, pmz_x, pmz_y)

## with one of the two precursor m/z being NA, the result are the same as
## with joinPeaks (with type = "left").
joinPeaksGnps(x, y, pmz_x, yPrecursorMz = NA)
}
\seealso{
\itemize{
\item \code{\link[=compareSpectra]{compareSpectra()}} for the function to calculate similarities between
spectra.
\item \code{\link[MsCoreUtils:gnps]{MsCoreUtils::gnps()}} in the \emph{MsCoreUtils} package for more information
on the GNPS similarity score.
}
}
\author{
Johannes Rainer, Michael Witting
}
