/****************************************************************************
 *                     Subassignment of a sparse vector                     *
 ****************************************************************************/
#include "SparseVec_subassignment.h"


static SEXPTYPE get_SV_subassign_Rtype(const SparseVec *sv,
		SEXP Rvector, const SparseVec *out_sv)
{
	SEXPTYPE Rtype = get_SV_Rtype(out_sv);
	if (sv->len != out_sv->len || get_SV_Rtype(sv) != Rtype)
		error("SparseArray internal error in "
		      "get_SV_subassign_Rtype():\n"
		      "    'sv' and 'out_sv' are incompatible");
	if (TYPEOF(Rvector) != Rtype)
		error("SparseArray internal error in "
		      "get_SV_subassign_Rtype():\n"
		      "    'Rvector' and 'out_sv' don't have the same type");
	return Rtype;
}


/****************************************************************************
 * Inline functions next_<type>_out()
 */

#define DEFINE_next_type_out_FUN(type)					      \
static inline int next_ ## type ## _out(				      \
		const SparseVec *sv1, const int *offs2, int n2,		      \
		const type *vals2, const int *selection2,		      \
		int *k1, int *k2, int *off, type *out_val)		      \
{									      \
	int ret = next_offset(sv1->nzoffs, get_SV_nzcount(sv1),		      \
			      offs2, n2, *k1, *k2, off);		      \
	if (ret == 1) {							      \
		*out_val = get_ ## type ## SV_nzval(sv1, *k1);		      \
		(*k1)++;						      \
	} else if (ret >= 2) {						      \
		int i = selection2 == NULL ? *k2 : selection2[*k2];	      \
		*out_val = vals2[i];					      \
		(*k2)++;						      \
		if (ret == 3)						      \
			(*k1)++;					      \
	}								      \
	return ret;							      \
}

DEFINE_next_type_out_FUN(int)
DEFINE_next_type_out_FUN(double)
DEFINE_next_type_out_FUN(Rcomplex)
DEFINE_next_type_out_FUN(Rbyte)

static inline int next_character_elt_out(
		const SparseVec *sv1, const int *offs2, int n2,
		SEXP Rvector, R_xlen_t block_offset, const int *selection,
		int *k1, int *k2, int *off, SEXP *out_val)
{
	int ret = next_offset(sv1->nzoffs, get_SV_nzcount(sv1),
			      offs2, n2, *k1, *k2, off);
	if (ret == 1) {
		*out_val = get_characterSV_nzval(sv1, *k1);
		(*k1)++;
	} else if (ret >= 2) {
		int i = selection == NULL ? block_offset + *k2 :
					    selection[*k2];
		*out_val = STRING_ELT(Rvector, i);
		(*k2)++;
		if (ret == 3)
			(*k1)++;
	}
	return ret;
}

static inline int next_list_elt_out(
		const SparseVec *sv1, const int *offs2, int n2,
		SEXP Rvector, R_xlen_t block_offset, const int *selection,
		int *k1, int *k2, int *off, SEXP *out_val)
{
	int ret = next_offset(sv1->nzoffs, get_SV_nzcount(sv1),
			      offs2, n2, *k1, *k2, off);
	if (ret == 1) {
		*out_val = get_listSV_nzval(sv1, *k1);
		(*k1)++;
	} else if (ret >= 2) {
		int i = selection == NULL ? block_offset + *k2 :
					    selection[*k2];
		*out_val = VECTOR_ELT(Rvector, i);
		(*k2)++;
		if (ret == 3)
			(*k1)++;
	}
	return ret;
}


/****************************************************************************
 * subassign_SV()
 */

#define DEFINE_subassign_typeSV_FUN(type)				   \
static int subassign_ ## type ## SV(					   \
		const SparseVec *sv1, const int *offs2, int n2,		   \
		const type *vals2, const int *selection2,		   \
		SparseVec *out_sv)					   \
{									   \
	type *out_nzvals = (type *) out_sv->nzvals;			   \
	type out_bg_val = out_sv->na_background ? type ## NA : type ## 0;  \
	out_sv->nzcount = 0;						   \
	int ret, k1 = 0, k2 = 0, off, neffrep = 0;			   \
	type out_val;							   \
	while ((ret = next_ ## type ## _out(sv1, offs2, n2,		   \
					    vals2, selection2,		   \
					    &k1, &k2, &off, &out_val)))	   \
	{								   \
		if (ret == 2) {						   \
			if (type ## _equal(out_val, out_bg_val)) 	   \
				continue;				   \
			neffrep++;					   \
		} else if (ret == 3) {					   \
			if (type ## _equal(out_val, out_bg_val)) {	   \
				neffrep++;				   \
				continue;				   \
			}						   \
			type v1 = get_ ## type ## SV_nzval(sv1, k1 - 1);   \
			if (!type ## _equal(v1, out_val))		   \
				neffrep++;				   \
		}							   \
		APPEND_TO_NZVALS_NZOFFS(out_val, off,			   \
			out_nzvals, out_sv->nzoffs, out_sv->nzcount);	   \
	}								   \
	return neffrep;							   \
}

DEFINE_subassign_typeSV_FUN(int)
DEFINE_subassign_typeSV_FUN(double)
DEFINE_subassign_typeSV_FUN(Rcomplex)

static int subassign_RbyteSV(
		const SparseVec *sv1, const int *offs2, int n2,
		const Rbyte *vals2, const int *selection2,
		SparseVec *out_sv)
{
	Rbyte *out_nzvals = (Rbyte *) out_sv->nzvals;
	out_sv->nzcount = 0;
	int ret, k1 = 0, k2 = 0, off, neffrep = 0;
	Rbyte out_val;
	while ((ret = next_Rbyte_out(sv1, offs2, n2,
				     vals2, selection2,
				     &k1, &k2, &off, &out_val)))
	{
		if (ret == 2) {
			if (out_val == Rbyte0)
				continue;  /* zero replaces zero */
			/* nonzero replaces zero */
			neffrep++;
		} else if (ret == 3) {
			if (out_val == Rbyte0) {
				/* zero replaces nonzero */
				neffrep++;
				continue;
			}
			/* nonzero replaces nonzero */
			Rbyte v1 = get_RbyteSV_nzval(sv1, k1 - 1);
			if (v1 != out_val)
				neffrep++;
		}
		APPEND_TO_NZVALS_NZOFFS(out_val, off,
			out_nzvals, out_sv->nzoffs, out_sv->nzcount);
	}
	return neffrep;
}

/* Note that when comparing CHARSXPs 'v1' and 'out_val' below (v1 != out_val),
   we compare their **addresses**, not their **values**.
   However, this is much faster, but also, and most importantly, it's
   equivalent to comparing their values. That's because CHARSXPs with the
   same value are expected to have the same address, thanks to R's global
   CHARSXP cache.
   In any case, even if 'v1 != out_val' were to produce false positives,
   it would not be such a big deal because the main reason for counting the
   number of **effective** replacements (neffrep) is to avoid copying an SVT
   leaf when a subassignment does not modify it (i.e. when 'neffrep == 0').
   So in the worst case, these false positives simply mean that we would
   still copy a leaf touched by the subassignment operation, even if the
   leaf is not modified by the subassignment. */
static int subassign_characterSV(
		const SparseVec *sv1, const int *offs2, int n2,
		SEXP Rvector, R_xlen_t block_offset, const int *selection,
		SparseVec *out_sv)
{
	SEXP out_nzvals = (SEXP) out_sv->nzvals;  /* STRSXP */
	out_sv->nzcount = 0;
	int ret, k1 = 0, k2 = 0, off, neffrep = 0;
	SEXP out_val;
	while ((ret = next_character_elt_out(sv1, offs2, n2,
					     Rvector, block_offset, selection,
					     &k1, &k2, &off, &out_val)))
	{
		if (ret == 2) {
			if (IS_BG_CHARSXP(out_val, out_sv->na_background))
				continue;
			neffrep++;
		} else if (ret == 3) {
			if (IS_BG_CHARSXP(out_val, out_sv->na_background)) {
				neffrep++;
				continue;
			}
			SEXP v1 = get_characterSV_nzval(sv1, k1 - 1);
			/* See note above about this comparison. */
			if (v1 != out_val)
				neffrep++;
		}
		SET_STRING_ELT(out_nzvals, out_sv->nzcount, out_val);
		out_sv->nzoffs[out_sv->nzcount] = off;
		out_sv->nzcount++;
	}
	return neffrep;
}

/* Note that when comparing VECSXP elements 'v1' and 'out_val' below
   (v1 != out_val), we compare their **addresses**, not their **values**.
   Comparing the values would be too costly. So yes, 'v1 != out_val' can
   produce false positives, but it's not a big deal because the main
   reason for counting the number of **effective** replacements (neffrep)
   is to avoid copying an SVT leaf when a subassignment does not modify
   it (i.e. when 'neffrep == 0').
   So in the worst case, these false positives simply mean that we will
   still copy a leaf touched by the subassignment operation, even if the
   leaf is not modified by the subassignment.
   However, comparing the addresses will still do a good job in a situation
   like:

       value <- subset_Array_by_Nindex(svt1, Nindex)
       svt2 <- subassign_Array_by_Nindex(svt1, Nindex, as.array(value))

   where no copy will be triggered ('svt2@SVT' will have the same address
   as 'svt1@SVT'). */
static int subassign_listSV(
		const SparseVec *sv1, const int *offs2, int n2,
		SEXP Rvector, R_xlen_t block_offset, const int *selection,
		SparseVec *out_sv)
{
	SEXP out_nzvals = (SEXP) out_sv->nzvals;  /* VECSXP */
	out_sv->nzcount = 0;
	int ret, k1 = 0, k2 = 0, off, neffrep = 0;
	SEXP out_val;
	while ((ret = next_list_elt_out(sv1, offs2, n2,
					Rvector, block_offset, selection,
					&k1, &k2, &off, &out_val)))
	{
		if (ret == 2) {
			if (out_val == R_NilValue)
				continue;
			neffrep++;
		} else if (ret == 3) {
			if (out_val == R_NilValue) {
				neffrep++;
				continue;
			}
			SEXP v1 = get_listSV_nzval(sv1, k1 - 1);
			/* See note above about this comparison. */
			if (v1 != out_val)
				neffrep++;
		}
		SET_VECTOR_ELT(out_nzvals, out_sv->nzcount, out_val);
		out_sv->nzoffs[out_sv->nzcount] = off;
		out_sv->nzcount++;
	}
	return neffrep;
}

static int subassign_SV(
		const SparseVec *sv1, const int *offs2, int n2,
		SEXP Rvector, R_xlen_t block_offset, const int *selection,
		SparseVec *out_sv)
{
	SEXPTYPE Rtype = get_SV_subassign_Rtype(sv1, Rvector, out_sv);
	switch (Rtype) {
	    case INTSXP: case LGLSXP:
		return subassign_intSV(sv1, offs2, n2,
				INTEGER(Rvector) + block_offset, selection,
				out_sv);
	    case REALSXP:
		return subassign_doubleSV(sv1, offs2, n2,
				REAL(Rvector) + block_offset, selection,
				out_sv);
	    case CPLXSXP:
		return subassign_RcomplexSV(sv1, offs2, n2,
				COMPLEX(Rvector) + block_offset, selection,
				out_sv);
	    case RAWSXP:
		return subassign_RbyteSV(sv1, offs2, n2,
				RAW(Rvector) + block_offset, selection,
				out_sv);
	    case STRSXP:
		return subassign_characterSV(sv1, offs2, n2,
				Rvector, block_offset, selection,
				out_sv);
	    case VECSXP:
		return subassign_listSV(sv1, offs2, n2,
				Rvector, block_offset, selection,
				out_sv);
	}
	error("SparseArray internal error in subassign_SV():\n"
	      "    'out_sv' of type \"%s\" not supported", type2char(Rtype));
	return 0;  /* will never reach this */
}


/****************************************************************************
 * _subassign_SV_with_Rvector_block()
 * _subassign_SV_with_Rvector_subset()
 */

/* 'sv->len' and 'out_sv->len' must be the same. 'sv' can be lacunar.
   'offs' must be an array of 'n' offsets (non-negative integers) that are
   strictly sorted (in ascending order). The last offset in the array must
   be < 'sv->len'.
   The block of 'n' elements in 'Rvector' that starts at offset 'block_offset'
   forms the replacement value (a.k.a. right value) of the subassignment
   operation. It can contain zeros.
   Returns the number of **effective** replacements, that is, the number of
   offsets for which the subassignment operation effectively modifies the
   original value. */
int _subassign_SV_with_Rvector_block(
		const SparseVec *sv, const int *offs, int n,
		SEXP Rvector, R_xlen_t block_offset, SparseVec *out_sv)
{
	return subassign_SV(sv, offs, n,
			    Rvector, block_offset, NULL, out_sv);
}

int _subassign_SV_with_Rvector_subset(
		const SparseVec *sv, const int *offs, int n,
		SEXP Rvector, const int *selection, SparseVec *out_sv)
{
	return subassign_SV(sv, offs, n,
			    Rvector, 0, selection, out_sv);
}


/****************************************************************************
 * subassign_full_<type>SV_with_Rvector_block()
 */

#define DEFINE_subassign_full_typeSV_with_Rvector_block_FUN(type)	  \
static int subassign_full_ ## type ## SV_with_Rvector_block(		  \
		const SparseVec *sv1,					  \
		const type *vals2, SparseVec *out_sv)			  \
{									  \
	type *out_nzvals = (type *) out_sv->nzvals;			  \
	type out_bg_val = out_sv->na_background ? type ## NA : type ## 0; \
	out_sv->nzcount = 0;						  \
	int k1 = 0, neffrep = 0;					  \
	for (int i = 0; i < out_sv->len; i++) {				  \
		type v2 = vals2[i];					  \
		if (k1 < get_SV_nzcount(sv1) && sv1->nzoffs[k1] == i) {	  \
			type v1 = get_ ## type ## SV_nzval(sv1, k1);	  \
			k1++;						  \
			if (type ## _equal(v2, out_bg_val)) {		  \
				neffrep++;				  \
				continue;				  \
			}						  \
			if (!type ## _equal(v1, v2))			  \
				neffrep++;				  \
		} else {						  \
			if (type ## _equal(v2, out_bg_val))		  \
				continue;				  \
			neffrep++;					  \
		}							  \
		APPEND_TO_NZVALS_NZOFFS(v2, i,				  \
			out_nzvals, out_sv->nzoffs, out_sv->nzcount);	  \
	}								  \
	return neffrep;							  \
}

DEFINE_subassign_full_typeSV_with_Rvector_block_FUN(int)
DEFINE_subassign_full_typeSV_with_Rvector_block_FUN(double)
DEFINE_subassign_full_typeSV_with_Rvector_block_FUN(Rcomplex)

static int subassign_full_RbyteSV_with_Rvector_block(const SparseVec *sv1,
		const Rbyte *vals2, SparseVec *out_sv)
{
	Rbyte *out_nzvals = (Rbyte *) out_sv->nzvals;
	out_sv->nzcount = 0;
	int k1 = 0, neffrep = 0;
	for (int i = 0; i < out_sv->len; i++) {
		Rbyte v2 = vals2[i];
		if (k1 < get_SV_nzcount(sv1) && sv1->nzoffs[k1] == i) {
			Rbyte v1 = get_RbyteSV_nzval(sv1, k1);
			k1++;
			if (v2 == Rbyte0) {
				/* zero replaces nonzero */
				neffrep++;
				continue;
			}
			/* nonzero replaces nonzero */
			if (v1 != v2)
				neffrep++;
		} else {
			if (v2 == Rbyte0)
				continue;  /* zero replaces zero */
			/* nonzero replaces zero */
			neffrep++;
		}
		APPEND_TO_NZVALS_NZOFFS(v2, i,
			out_nzvals, out_sv->nzoffs, out_sv->nzcount);
	}
	return neffrep;
}

/* Note that when comparing CHARSXPs 'v1' and 'v2' below (v1 != v2), we
   compare their **addresses**, not their **values**.
   See note for subassign_characterSV() above for more information. */
static int subassign_full_characterSV_with_Rvector_block(const SparseVec *sv1,
		SEXP Rvector, R_xlen_t block_offset, SparseVec *out_sv)
{
	SEXP out_nzvals = (SEXP) out_sv->nzvals;  /* STRSXP */
	out_sv->nzcount = 0;
	int k1 = 0, neffrep = 0;
	for (int i = 0; i < out_sv->len; i++) {
		SEXP v2 = STRING_ELT(Rvector, block_offset + i);
		if (k1 < get_SV_nzcount(sv1) && sv1->nzoffs[k1] == i) {
			SEXP v1 = get_characterSV_nzval(sv1, k1);
			k1++;
			if (IS_BG_CHARSXP(v2, out_sv->na_background)) {
				neffrep++;
				continue;
			}
			/* See note above about this comparison. */
			if (v1 != v2)
				neffrep++;
		} else {
			if (IS_BG_CHARSXP(v2, out_sv->na_background))
				continue;
			neffrep++;
		}
		SET_STRING_ELT(out_nzvals, out_sv->nzcount, v2);
		out_sv->nzoffs[out_sv->nzcount] = i;
		out_sv->nzcount++;
	}
	return neffrep;
}

/* Note that when comparing VECSXP elements 'v1' and 'v2' below (v1 != v2),
   we compare their **addresses**, not their **values**.
   See note for subassign_listSV() above for more information. */
static int subassign_full_listSV_with_Rvector_block(const SparseVec *sv1,
		SEXP Rvector, R_xlen_t block_offset, SparseVec *out_sv)
{
	SEXP out_nzvals = (SEXP) out_sv->nzvals;  /* VECSXP */
	out_sv->nzcount = 0;
	int k1 = 0, neffrep = 0;
	for (int i = 0; i < out_sv->len; i++) {
		SEXP v2 = VECTOR_ELT(Rvector, block_offset + i);
		if (k1 < get_SV_nzcount(sv1) && sv1->nzoffs[k1] == i) {
			SEXP v1 = get_listSV_nzval(sv1, k1);
			k1++;
			if (v2 == R_NilValue) {
				neffrep++;
				continue;
			}
			/* See note above about this comparison. */
			if (v1 != v2)
				neffrep++;
		} else {
			if (v2 == R_NilValue)
				continue;
			neffrep++;
		}
		SET_VECTOR_ELT(out_nzvals, out_sv->nzcount, v2);
		out_sv->nzoffs[out_sv->nzcount] = i;
		out_sv->nzcount++;
	}
	return neffrep;
}


/****************************************************************************
 * _subassign_full_SV_with_Rvector_block()
 */

/* Note that the content of input SparseVec 'sv' is used only to compute the
   number of **effective** replacements. In particular, it has NO impact on
   the content that gets written to 'out_sv'.
   In other words, _subassign_full_SV_with_Rvector_block() is equivalent to:

     _write_Rvector_block_to_SV(Rvector, block_offset,
                                NULL, out_sv->len, out_sv)

   except that it uses the content of 'sv' to compute the number
   of **effective** replacements (which it returns).

   'sv->len' and 'out_sv->len' must be the same. 'sv' can be lacunar.
   The block of 'n' elements in 'Rvector' that starts at offset 'block_offset'
   forms the replacement value (a.k.a. right value) of the subassignment
   operation. It can contain zeros.
   Returns the number of **effective** replacements, that is, the number of
   offsets for which the subassignment operation effectively modifies the
   original value. */
int _subassign_full_SV_with_Rvector_block(const SparseVec *sv,
		SEXP Rvector, R_xlen_t block_offset, SparseVec *out_sv)
{
	SEXPTYPE Rtype = get_SV_subassign_Rtype(sv, Rvector, out_sv);
	switch (Rtype) {
	    case INTSXP: case LGLSXP:
		return subassign_full_intSV_with_Rvector_block(sv,
				INTEGER(Rvector) + block_offset, out_sv);
	    case REALSXP:
		return subassign_full_doubleSV_with_Rvector_block(sv,
				REAL(Rvector) + block_offset, out_sv);
	    case CPLXSXP:
		return subassign_full_RcomplexSV_with_Rvector_block(sv,
				COMPLEX(Rvector) + block_offset, out_sv);
	    case RAWSXP:
		return subassign_full_RbyteSV_with_Rvector_block(sv,
				RAW(Rvector) + block_offset, out_sv);
	    case STRSXP:
		return subassign_full_characterSV_with_Rvector_block(sv,
				Rvector, block_offset, out_sv);
	    case VECSXP:
		return subassign_full_listSV_with_Rvector_block(sv,
				Rvector, block_offset, out_sv);
	}
	error("SparseArray internal error in "
	      "_subassign_full_SV_with_Rvector_block():\n"
	      "    'out_sv' of type \"%s\" not supported", type2char(Rtype));
	return 0;  /* will never reach this */
}

