## ----setup, include=TRUE------------------------------------------------------
# Standard setup chunk
knitr::opts_chunk$set(echo = TRUE, collapse = TRUE)
# Load libraries required for the vignette to build
library(PMScanR)
library(ggseqlogo)
library(seqinr)
library(plotly)

## ----installation-loading, eval=FALSE-----------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("PMScanR")
# 
# library(PMScanR)

## ----run-shiny-app, eval=FALSE------------------------------------------------
# # This command launches the interactive Shiny app
# runPMScanRShiny()

## ----set-working-directory----------------------------------------------------
# Setting working directory is user-specific,
# setwd("disc:/your/path/to/working/directory")

## ----load-example-fasta-------------------------------------------------------
# Load example FASTA file included with the package
fasta_file <- system.file("extdata", "hemoglobins.fasta", package = "PMScanR")

## ----runPsScan, eval=FALSE----------------------------------------------------
# # This command is not evaluated in the vignette as it requires an external
# # dependency (Perl) and can be time-consuming.
# runPsScan(in_file = fasta_file, out_format = 'gff', out_file = "results_pfscan.gff")

## ----read-example-psa---------------------------------------------------------
# Load the path to an example PSA output file
motifs_psa_file <- system.file("extdata", "out_Hb_psa.txt", package = "PMScanR")

# Use readPsa to convert the file to a GFF-like data frame
gff_data <- readPsa(motifs_psa_file)

# Display the first few rows of the resulting data frame
head(gff_data)

## ----convert-gff-to-matrix----------------------------------------------------
# The 'gff_data' is used from the previous chunk
motif_matrix <- gff2matrix(gff_data)

# Display the first few rows of the resulting matrix
head(motif_matrix)

## ----generate-heatmap-1-------------------------------------------------------
# Generate a standard heatmap from the motif_matrix
heatmap1 <- matrix2hm(input = motif_matrix)
heatmap1

## ----generate-heatmap-2, eval=FALSE-------------------------------------------
# # Generate a square heatmap from the motif_matrix
# heatmap2 <- matrixToSquareHeatmap(input = motif_matrix)
# heatmap2

## ----extract-segments-from-fasta, fig.show='hold'-----------------------------
# Read the FASTA file into a list of sequences
sequences <- seqinr::read.fasta(file = fasta_file, seqtype = "AA")

# Extract segments from position 10 to 20 from all sequences
segments <- extractSegments(sequences, from = 10, to = 20)

# Generate the sequence logo from the extracted segments
ggseqlogo::ggseqlogo(unlist(segments), seq_type = "aa")

## ----extract-motifs-from-psa, fig.show='hold'---------------------------------
# Extract all protein motifs into a list
protein_motifs <- extractProteinMotifs(motifs_psa_file)

# Generate the sequence logo from the extracted protein motifs
ggseqlogo::ggseqlogo(protein_motifs$PS60007, seq_type='aa') # Sequence logo for motif 'PS60007' (example motif ID)
ggseqlogo::ggseqlogo(protein_motifs[1], seq_type='aa')       # Sequence logo for the first motif in the list
ggseqlogo::ggseqlogo(protein_motifs[5], seq_type='aa')       # Sequence logo for the fifth motif in the list

## ----generate-pie-chart, eval=TRUE--------------------------------------------
# The 'gff_data'  from a previous chunk is used here
pie_chart <- freqPie(gff_data)
print(pie_chart)

## ----session-info-------------------------------------------------------------
sessionInfo()

