% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_spike_percentage_list.R
\name{calculate_spike_percentage_list}
\alias{calculate_spike_percentage_list}
\title{Calculate Spike-in Percentage for Specified Taxa}
\usage{
calculate_spike_percentage_list(
  obj,
  merged_spiked_species,
  output_path = NULL,
  passed_range = c(0.1, 11)
)
}
\arguments{
\item{obj}{A \code{phyloseq} or \code{TreeSummarizedExperiment} object.}

\item{merged_spiked_species}{Character vector or list of spike-in species names (at the \code{Species} level).}

\item{output_path}{Optional. Character string specifying the output path (DOCX).
If \code{NULL} (default) results are not saved.}

\item{passed_range}{Numeric vector of length 2 specifying the accepted percentage range.
Default is \code{c(0.1, 11)}.}
}
\value{
A \code{data.frame} with:
\itemize{
\item Sample
\item Total_Reads
\item Total_Reads_spiked
\item Percentage (of spike-in reads)
\item Result ("passed"/"failed")
}
}
\description{
Computes the percentage of reads attributed to specified spike-in taxa in a
\code{phyloseq} or \code{TreeSummarizedExperiment} object. The function merges spike-in taxa,
computes percentages, classifies samples into "passed" or "failed" based on a
user-defined threshold, and optionally exports DOCX and CSV reports.
}
\details{
The function automatically detects spike-in OTUs based on the \code{Species} column in
the taxonomy table. It works with both \code{phyloseq} and \code{TreeSummarizedExperiment}
objects and produces QC diagnostics commonly required for spike-in based absolute
quantification workflows.
}
\section{Notes}{

\itemize{
\item Assumes the taxonomy table contains a column named \code{Species}.
\item Supports both \code{phyloseq} and \code{TreeSummarizedExperiment} objects.
}
}

\examples{
if (requireNamespace("DspikeIn", quietly = TRUE)) {
  # Load example phyloseq and TSE objects
  data("physeq", package = "DspikeIn")
  data("tse", package = "DspikeIn")

  # Define merged spike-in species list
  spiked_species_list <- c(
    "Pseudomonas aeruginosa",
    "Escherichia coli",
    "Clostridium difficile"
  )

  # Create temporary output paths
  temp_docx <- file.path(tempdir(), "merged_result.docx")
  temp_csv <- sub(".docx", ".csv", temp_docx)

  # --- Phyloseq Example ---
  result_physeq <- calculate_spike_percentage_list(
    obj = physeq,
    merged_spiked_species = spiked_species_list,
    output_path = temp_docx,
    passed_range = c(0.1, 10)
  )
  print(result_physeq)

  # --- TSE Example ---
  result_tse <- calculate_spike_percentage_list(
    obj = tse,
    merged_spiked_species = spiked_species_list,
    output_path = temp_docx,
    passed_range = c(0.1, 10)
  )
  print(result_tse)

  # Clean up
  if (file.exists(temp_docx)) unlink(temp_docx, force = TRUE)
  if (file.exists(temp_csv)) unlink(temp_csv, force = TRUE)
}
}
