% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_spikeIn_factors.R
\name{calculate_spikeIn_factors}
\alias{calculate_spikeIn_factors}
\title{Calculate Scaling Factors for Spiked Species in Phyloseq or TSE Object}
\usage{
calculate_spikeIn_factors(
  obj,
  spiked_cells,
  merged_spiked_species,
  output_path = NULL
)
}
\arguments{
\item{obj}{A \code{phyloseq} or \code{TreeSummarizedExperiment} object containing microbiome data.}

\item{spiked_cells}{A numeric value for the number of spiked cells per unit volume.}

\item{merged_spiked_species}{A character vector of spiked taxon names (species or genus).}

\item{output_path}{Optional directory path to save intermediate files (default is NULL).}
}
\value{
A list with:
\item{scaling_factors}{Named numeric vector of scaling factors per sample.}
\item{filtered_obj}{Input object with spike-in taxa removed.}
\item{spiked_species_reads}{Data frame with spike-in reads per sample.}
\item{total_reads}{Data frame with total reads per sample.}
\item{spiked_species_merged}{The merged spike-in taxa as a phyloseq object.}
\item{tree}{Original phylogenetic tree, if available.}
}
\description{
Calculates scaling factors for specified spike-in species or genera in
a \code{phyloseq} or \code{TreeSummarizedExperiment (TSE)} object. It supports genus/species-level detection,
removes spike-ins, merges them, computes scaling factors, and returns a bias-corrected absolute count matrix.
The function automatically handles:Species or genus-level spike-in identification, Safe tree and taxonomy synchronization,
Volume-based scaling (via the spiked.volume field in metadata),
Optional export of intermediate results for traceability (Total_Reads.csv, Spiked_Reads.csv, Scaling_Factors.csv).
}
\examples{
if (requireNamespace("DspikeIn", quietly = TRUE) &&
  requireNamespace("phyloseq", quietly = TRUE)) {
  data("physeq_16SOTU", package = "DspikeIn")

  spiked_cells <- 1847
  species_name <- spiked_species <- c("Tetragenococcus_halophilus", "Tetragenococcus_sp.")
  merged_spiked_species <- "Tetragenococcus_halophilus"

  # --- Phyloseq Example ---
  spiked_16S_OTU <- phyloseq::subset_samples(physeq_16SOTU, spiked.volume \%in\% c("2", "1"))
  temp_output_file <- file.path(tempdir(), "merged_physeq_sum.rds")
  output_dir <- file.path(tempdir(), "spikeIn_factors_output")

  merged_physeq_sum <- Pre_processing_species(
    spiked_16S_OTU,
    species_name = merged_spiked_species,
    merge_method = "sum",
    output_file = temp_output_file
  )

  result_physeq <- calculate_spikeIn_factors(
    merged_physeq_sum,
    spiked_cells = spiked_cells,
    merged_spiked_species = merged_spiked_species,
    output_path = output_dir
  )

  print(result_physeq$scaling_factors)

  if (file.exists(temp_output_file)) unlink(temp_output_file, force = TRUE)
  if (dir.exists(output_dir)) unlink(output_dir, recursive = TRUE, force = TRUE)

  # --- TSE Example ---
  tse_data <- convert_phyloseq_to_tse(physeq_16SOTU)
  merged_tse_sum <- Pre_processing_species(
    tse_data,
    species_name = merged_spiked_species,
    merge_method = "sum"
  )

  result_tse <- calculate_spikeIn_factors(
    merged_tse_sum,
    spiked_cells = spiked_cells,
    merged_spiked_species = merged_spiked_species
  )

  print(result_tse$scaling_factors)

  # --- Final cleanup of any extra leftover RDS files ---
  leftover_rds <- list.files(tempdir(), pattern = "merged_physeq.*\\\\.rds$", full.names = TRUE)
  file.remove(leftover_rds[file.exists(leftover_rds)])
}
}
