## ----echo=TRUE, message=FALSE, warning=FALSE, eval=FALSE----------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("looking4clusters")

## ----echo=TRUE, message=FALSE, warning=FALSE, eval=TRUE-----------------------
library("looking4clusters")

## ----echo=TRUE, message=FALSE, warning=FALSE----------------------------------
# Load sample data from scRNAseq package
library(scRNAseq)
sce <- ReprocessedAllenData("tophat_counts")
counts <- assay(sce, "tophat_counts")

# Perform dimensional reduction and an automatic cluster identification
obj <- looking4clusters(t(counts), groups=colData(sce)[,'dissection_s'])

# Output interactive visualization
l4chtml(obj)

## ----echo=FALSE, out.width='100%'---------------------------------------------
knitr::include_graphics("auto.png")

## ----echo=TRUE, message=FALSE, warning=FALSE----------------------------------
# Create a new looking for cluster object
obj <- looking4clusters(t(counts), running_all=FALSE)

# Add a sample clasification from input data
groups <- colData(sce)[,'dissection_s']
obj <- addcluster(obj, groups, myGroups=TRUE)

# Perform a PCA and TSNE and add to the object as a dimensional reduction layout
libsizes <- colSums(counts)
size.factors <- libsizes/mean(libsizes)
logcounts(sce) <- log2(t(t(counts)/size.factors) + 1)

pca_data <- prcomp(t(logcounts(sce)), rank=50)
obj <- addreduction(obj, pca_data$x[,1:2], "PCA")

library(Rtsne)
tsne_data <- Rtsne(pca_data$x[,1:50], pca = FALSE)
obj <- addreduction(obj, tsne_data$Y, "TSNE")

# Output interactive visualization
l4chtml(obj)

## ----echo=FALSE, out.width='100%'---------------------------------------------
knitr::include_graphics("manual.png")

## ----echo=TRUE, message=FALSE, warning=FALSE----------------------------------
# Adding PCA and TSNE to the object
reducedDims(sce) <- list(PCA=pca_data$x, TSNE=tsne_data$Y)

# Create a looking4clusters object from a SingleCellExperiment object
obj <- looking4clusters(sce, groups="dissection_s")

# Output interactive visualization
l4chtml(obj)

## ----echo=FALSE, out.width='100%'---------------------------------------------
knitr::include_graphics("sce.png")

## ----echo=TRUE, message=FALSE, warning=FALSE, eval=FALSE----------------------
# library(Seurat)
# library(Matrix)
# 
# # Load sample data from ZilionisLungData
# lung <- ZilionisLungData()
# immune <- lung$Used & lung$used_in_NSCLC_immune
# lung <- lung[,immune]
# lung <- lung[1:10000,1:1000]
# 
# exp_mat <- Matrix(counts(lung),sparse = TRUE)
# colnames(exp_mat) <- paste0(colnames(exp_mat), seq(1,ncol(exp_mat)))
# 
# # Create a new Seurat object
# seurat_object <- CreateSeuratObject(counts = exp_mat,
#                                     project = "Zilionis_immune")
# 
# # Seurat data processing steps
# seurat_object <- NormalizeData(seurat_object)
# seurat_object <- ScaleData(seurat_object, features = rownames(seurat_object))
# 
# seurat_object <- FindVariableFeatures(seurat_object)
# seurat_object <- RunPCA(seurat_object,
#     features = VariableFeatures(object = seurat_object))
# 
# # Create a looking4clusters object from a Seurat object
# obj <- looking4clusters(seurat_object)
# 
# # Output interactive visualization
# l4chtml(obj)

## ----echo=FALSE, out.width='100%'---------------------------------------------
knitr::include_graphics("seurat.png")

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

