% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods-XCMSnExp.R
\name{chromatogram,XCMSnExp-method}
\alias{chromatogram,XCMSnExp-method}
\alias{chromatogram}
\title{Extracting chromatograms}
\usage{
\S4method{chromatogram}{XCMSnExp}(
  object,
  rt,
  mz,
  aggregationFun = "sum",
  missing = NA_real_,
  msLevel = 1L,
  BPPARAM = bpparam(),
  adjustedRtime = hasAdjustedRtime(object),
  filled = FALSE,
  include = c("apex_within", "any", "none"),
  ...
)
}
\arguments{
\item{object}{Either a \link{OnDiskMSnExp} or \link{XCMSnExp} object from which the
chromatograms should be extracted.}

\item{rt}{\code{numeric(2)} or two-column \code{matrix} defining the lower
and upper boundary for the retention time range(s). If not specified,
the full retention time range of the original data will be used.}

\item{mz}{\code{numeric(2)} or two-column \code{matrix} defining the lower
and upper mz value for the MS data slice(s). If not specified, the
chromatograms will be calculated on the full mz range.}

\item{aggregationFun}{\code{character(1)} specifying the function to be used to
aggregate intensity values across the mz value range for the same
retention time. Allowed values are \code{"sum"} (the default), \code{"max"},
\code{"mean"} and \code{"min"}.}

\item{missing}{\code{numeric(1)} allowing to specify the intensity value to
be used if for a given retention time no signal was measured within the
mz range of the corresponding scan. Defaults to \code{NA_real_} (see also
Details and Notes sections below). Use \code{missing = 0} to resemble the
behaviour of the \code{getEIC} from the \emph{old} user interface.}

\item{msLevel}{\code{integer(1)} specifying the MS level from which the
chromatogram should be extracted. Defaults to \code{msLevel = 1L}.}

\item{BPPARAM}{Parallelisation backend to be used, which will
depend on the architecture. Default is
\code{BiocParallel::bparam()}.}

\item{adjustedRtime}{For \verb{chromatogram,XCMSnExp}: whether the
adjusted (\code{adjustedRtime = TRUE}) or raw retention times
(\code{adjustedRtime = FALSE}) should be used for filtering and returned
in the resulting \link{MChromatograms} object. Adjusted
retention times are used by default if available.}

\item{filled}{\code{logical(1)} whether filled-in peaks should also be
returned. Defaults to \code{filled = FALSE}, i.e. returns only detected
chromatographic peaks in the result object.}

\item{include}{\code{character(1)} defining which chromatographic peaks should be
returned. Supported are \code{include = "apex_within"} (the default) which
returns chromatographic peaks that have their apex within the \code{mz} \code{rt}
range, \code{include = "any"} to return all chromatographic peaks which
m/z and rt ranges overlap the \code{mz} and \code{rt} or \code{include = "none"} to
not include any chromatographic peaks.}

\item{...}{optional parameters - currently ignored.}
}
\value{
\code{chromatogram} returns a \link{XChromatograms} object with
the number of columns corresponding to the number of files in
\code{object} and number of rows the number of specified ranges (i.e.
number of rows of matrices provided with arguments \code{mz} and/or
\code{rt}). All chromatographic peaks with their apex position within the
m/z and retention time range are also retained as well as all feature
definitions for these peaks.
}
\description{
\code{chromatogram}: extract chromatographic data (such as an extracted ion
chromatogram, a base peak chromatogram or total ion chromatogram) from
an \link{OnDiskMSnExp} or \link{XCMSnExp} objects. See also the help page of the
\code{chromatogram} function in the \code{MSnbase} package.
}
\details{
Arguments \code{rt} and \code{mz} allow to specify the MS data slice (i.e. the m/z
range and retention time window) from which the chromatogram should be
extracted. These parameters can be either a \code{numeric} of length 2 with the
lower and upper limit, or a \code{matrix} with two columns with the lower and
upper limits to extract multiple EICs at once.
The parameter \code{aggregationSum} allows to specify the function to be
used to aggregate the intensities across the m/z range for the same
retention time. Setting \code{aggregationFun = "sum"} would e.g. allow
to calculate the \strong{total ion chromatogram} (TIC),
\code{aggregationFun = "max"} the \strong{base peak chromatogram} (BPC).

If for a given retention time no intensity is measured in that spectrum a
\code{NA} intensity value is returned by default. This can be changed with the
parameter \code{missing}, setting \code{missing = 0} would result in a \code{0} intensity
being returned in these cases.
}
\note{
For \link{XCMSnExp} objects, if adjusted retention times are
available, the \code{chromatogram} method will by default report
and use these (for the subsetting based on the provided parameter
\code{rt}). This can be changed by setting \code{adjustedRtime = FALSE}.
}
\examples{

## Load a test data set with identified chromatographic peaks
library(MSnbase)
data(faahko_sub)
## Update the path to the files for the local system
dirname(faahko_sub) <- system.file("cdf/KO", package = "faahKO")

## Disable parallel processing for this example
register(SerialParam())

## Extract the ion chromatogram for one chromatographic peak in the data.
chrs <- chromatogram(faahko_sub, rt = c(2700, 2900), mz = 335)

chrs

## Identified chromatographic peaks
chromPeaks(chrs)

## Plot the chromatogram
plot(chrs)

## Extract chromatograms for multiple ranges.
mzr <- matrix(c(335, 335, 344, 344), ncol = 2, byrow = TRUE)
rtr <- matrix(c(2700, 2900, 2600, 2750), ncol = 2, byrow = TRUE)
chrs <- chromatogram(faahko_sub, mz = mzr, rt = rtr)

chromPeaks(chrs)

plot(chrs)

## Get access to all chromatograms for the second mz/rt range
chrs[1, ]

## Plot just that one
plot(chrs[1, , drop = FALSE])
}
\seealso{
\link{XCMSnExp} for the data object.
\link{Chromatogram} for the object representing chromatographic data.

\if{html}{\out{<div class="sourceCode">}}\preformatted{[XChromatograms] for the object allowing to arrange
multiple [XChromatogram] objects.

[plot] to plot a [XChromatogram] or [MChromatograms] objects.

`as` (`as(x, "data.frame")`) in `MSnbase` for a method to extract
the MS data as `data.frame`.
}\if{html}{\out{</div>}}
}
\author{
Johannes Rainer
}
