% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/testClusterPos.R
\name{testClusterPos}
\alias{testClusterPos}
\title{Test positional bias motifs within a cluster}
\usage{
testClusterPos(
  x,
  stringset,
  binwidth = 10,
  abs = FALSE,
  rc = TRUE,
  min_score = "80\%",
  break_ties = "all",
  alt = c("greater", "less", "two.sided"),
  sort_by = c("p", "none"),
  mc.cores = 1,
  ...
)
}
\arguments{
\item{x}{A Position Weight Matrix, universalmotif object or list thereof.
Alternatively can be a single DataFrame or list of DataFrames as returned
by \link{getClusterMatches} with \code{best_only = TRUE}}

\item{stringset}{An XStringSet. Not required if matches are supplied as x}

\item{binwidth}{Width of bins across the range to group data into}

\item{abs}{Use absolute positions around zero to find symmetrical enrichment}

\item{rc}{logical(1) Also find matches using the reverse complement of each
PWM}

\item{min_score}{The minimum score to return a match}

\item{break_ties}{Choose how to resolve matches with tied scores}

\item{alt}{Alternative hypothesis for the binomial test}

\item{sort_by}{Column to sort results by}

\item{mc.cores}{Passed to \link[parallel]{mclapply}}

\item{...}{Passed to \link[Biostrings]{matchPWM}}
}
\value{
A data.frame with columns \code{start}, \code{end}, \code{centre}, \code{width}, \code{total_matches},
\code{matches_in_region}, \code{expected}, \code{enrichment}, \code{prop_total}, \code{p}
and \code{consensus_motif}
The total matches represent the total number of matches within the set of
sequences, whilst the number observed in the final region are also given,
along with the proportion of the total this represents.
Enrichment is simply the ratio of observed to expected based on the
expectation of the null hypothesis

The consensus motif across all matches is returned as a Position Frequency
Matrix (PFM) using \link[Biostrings]{consensusMatrix}.
}
\description{
Test positional bias for all motifs within a given cluster
}
\details{
This is a reimplementation of \link{testMotifPos} for sets of motifs which
have been clustered for similarity.
The positions test the bias of any motifs within the cluster given that
overlapping matches are only counted once, and with the match retained being
the one with the highest relative score.

It should also be noted that some motif clusters will contain PWMs of
varying length. When finding positional bias, the widest motif is taken as
the width for all, and any matches from narrower motifs outside of the range
allowed by wider motifs are discarded. This reduction in signal will make a
small difference in the outer bins, but is not considered to be problematic
for the larger analysis.
}
\examples{
## Load the example PWM
data("ex_pfm")
## Load the example sequences
data("ar_er_seq")

## Cluster the motifs
cl <- list(A = ex_pfm[1], B = ex_pfm[2:3])

## Get the best match and use this data
matches <- getClusterMatches(cl, ar_er_seq, best_only = TRUE)
## Test for enrichment in any position
testClusterPos(matches)

## Or just pass the clustered matrices
## Here we've set abs = TRUE to test absolute distance from the centre
testClusterPos(cl, ar_er_seq, abs = TRUE, binwidth = 10)

}
